#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Simulateur interactif de transition des pensions - Version 4.3

Interface graphique permettant de :
- Simulation automatique au démarrage avec valeurs par défaut
- Mode lecture seule par défaut avec checkbox pour éditer
- Mise à jour automatique de la simulation quand une valeur change
- Indicateurs de couleur (gris=lecture seule, rouge pastel=éditable)
- Texte vert pour valeurs modifiées, noir pour originales
- Bouton reset à côté de chaque valeur
- Fenêtre agrandie avec checkbox "Live" pour mise à jour auto
- Support multi-devises et écrans haute résolution
- Panneau paramètres FIXE, panneau aide scrollable
- Menu pour ajuster la taille de police avec scaling global
- Sauvegarde graphiques SVG/PNG avec noms intelligents
- Codes 4 lettres pour champs et paramètres
- Scaling global de l'interface (fenêtre, dialogues, menus)
- Internationalisation (français/anglais)
- Scroll et pan sur les graphiques
"""

import tkinter as tk
from tkinter import ttk, filedialog, messagebox, scrolledtext
import sys
import os
from pathlib import Path
import configparser
import yaml
import numpy as np
import shutil
import json
import io
import subprocess
import platform
from PIL import Image, ImageTk

# Ajouter le chemin vers le module de simulation
SCRIPT_DIR = Path(__file__).parent
CODE_DIR = SCRIPT_DIR.parent / "code"
sys.path.insert(0, str(CODE_DIR))

# Taille de police de base (ajustable via menu)
BASE_FONT_SIZE = 14  # Taille par défaut

# Variable globale pour le facteur de scaling
SCALE = 1.0

def init_scaling():
    """Initialise le scaling avec des valeurs par défaut plus grandes"""
    global SCALE
    try:
        import scaling
        SCALE = scaling.auto_scaling()
        # Forcer au minimum 1.25 pour les écrans modernes
        if SCALE < 1.25:
            SCALE = 1.25
    except ImportError:
        print("[GUI] Module scaling non trouvé, utilisation de SCALE=1.25")
        SCALE = 1.25
    return SCALE

SCALE = init_scaling()

try:
    from transition_pensions import (
        Parametres, SimulateurTransition, charger_configuration
    )
except ImportError as e:
    print(f"Erreur d'import: {e}")
    print("Assurez-vous que transition_pensions.py est dans le dossier 'code'")
    sys.exit(1)

# Import du registre des métriques pour les métriques indexées
try:
    from metrics_registry import calculer_toutes_metriques_indexees
except ImportError:
    # Fallback si le module n'existe pas encore
    def calculer_toutes_metriques_indexees(resultats):
        return resultats

# Import du mode comparaison
try:
    from comparison_mode import (
        ComparisonConfig, ComparisonGridPanel, get_available_scenarios
    )
    COMPARISON_MODE_AVAILABLE = True
except ImportError:
    COMPARISON_MODE_AVAILABLE = False

# Import matplotlib avec backend tkinter
import matplotlib
matplotlib.use('TkAgg')
import matplotlib.pyplot as plt
from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg
from matplotlib.figure import Figure


# === SYSTÈME D'INTERNATIONALISATION ===
LANG_DIR = SCRIPT_DIR / "lang"
CURRENT_LANG = "fr"  # Langue par défaut
TRANSLATIONS = {}

def load_language(lang_code):
    """Charge un fichier de langue JSON"""
    global TRANSLATIONS, CURRENT_LANG
    lang_file = LANG_DIR / f"{lang_code}.json"
    if lang_file.exists():
        try:
            with open(lang_file, 'r', encoding='utf-8') as f:
                TRANSLATIONS = json.load(f)
                CURRENT_LANG = lang_code
                return True
        except Exception as e:
            print(f"Erreur chargement langue {lang_code}: {e}")
    return False

def get_available_languages():
    """Retourne la liste des langues disponibles"""
    langs = []
    if LANG_DIR.exists():
        for f in LANG_DIR.glob("*.json"):
            try:
                with open(f, 'r', encoding='utf-8') as file:
                    data = json.load(file)
                    langs.append((f.stem, data.get('lang_name', f.stem)))
            except:
                pass
    return langs

def t(key, **kwargs):
    """Récupère une traduction par clé (ex: 'menu.file')"""
    keys = key.split('.')
    value = TRANSLATIONS
    for k in keys:
        if isinstance(value, dict) and k in value:
            value = value[k]
        else:
            return key  # Retourne la clé si non trouvée
    # Remplacer les placeholders {name}
    if isinstance(value, str) and kwargs:
        for k, v in kwargs.items():
            value = value.replace('{' + k + '}', str(v))
    return value

# Charger la langue par défaut
load_language(CURRENT_LANG)


# === CONSTANTES ===
COLORS = {
    'primary': '#93c5fd',
    'secondary': '#c4b5fd',
    'success': '#86efac',
    'warning': '#fdba74',
    'danger': '#fca5a5',
    'neutral': '#d1d5db',
    'dark': '#374151',
    'readonly_bg': '#e5e7eb',      # Gris pour lecture seule
    'editable_bg': '#fecaca',       # Rouge pastel pour éditable
    'modified_fg': '#16a34a',       # Vert pour valeur modifiée
    'original_fg': '#000000',       # Noir pour valeur originale
}

CONFIG_DIR = SCRIPT_DIR.parent / "configurations"
GRAPHIQUES_DIR = SCRIPT_DIR / "graphiques"
TEMPLATE_FILE = GRAPHIQUES_DIR / "_template.yaml"

# Codes 4 lettres pour les champs (plus facile à taper)
FIELD_CODES = {
    'ANNE': 'annee',
    'PIBS': 'pib',
    'DIFF': 'differentiel_pct',
    'INTR': 'interets_totaux',
    'DPUB': 'dette_publique',
    'DTRA': 'dette_transition',
    'DTOT': 'dette_publique_totale',
    'DPPC': 'dette_publique_pct',
    'DTPC': 'dette_transition_pct',
    'TTPC': 'dette_publique_totale_pct',
    'DIMP': 'dette_implicite',
    'DIPC': 'dette_implicite_pct',
    'FLUX': 'flux_pensions',
    'NRET': 'nombre_retraites',
    'REQR': 'retraites_equiv_repartition',
    # Métriques indexées (base 100) pour comparaison
    'PIBI': 'pib_indexe',
    'DPBI': 'dette_publique_indexee',
    'DTBI': 'dette_totale_indexee',
    'DIBI': 'dette_implicite_indexee',
    'INBI': 'interets_indexes',
    # Champs salaires
    'SBRT': 'salaire_brut',
    'IMPS': 'impact_nouveau_systeme',
    'GTAX': 'gain_taxes_indirectes',
    'EFNT': 'effet_net_total',
}

# Codes 4 lettres pour les paramètres modifiables
PARAM_CODES = {
    'PIBI': 'pib_initial',
    'DPBI': 'dette_publique_initiale',
    'TCBS': 'taux_croissance_base',
    'DIBI': 'dette_implicite_initiale',
    'RSOL': 'reduction_solidaire',
    'PMOY': 'pension_moyenne_annuelle',
    'PRIV': 'privatisations_totales',
    'DFIN': 'differentiel_initial',
    'DDEC': 'duree_decroissance_differentiel',
    'TFLT': 'taux_flat_tax',
    'ABAT': 'abattement_forfaitaire',
    'ASAN': 'assurance_sante',
    'ACHO': 'assurance_chomage',
    'APEN': 'assurance_pension',
    'AEDU': 'assurance_education',
    'SBMP': 'surplus_budgetaire_minimal_pct_pib',
    'SMDT': 'surplus_max_pour_dette_transition_pct',
}

# Inverse: nom_paramètre -> code
PARAM_NAMES_TO_CODES = {v: k for k, v in PARAM_CODES.items()}

def resolve_field_code(code_or_name):
    """Convertit un code 4 lettres en nom de champ, ou retourne le nom tel quel"""
    return FIELD_CODES.get(code_or_name.upper(), code_or_name) if code_or_name else code_or_name

def get_yaml_value(obj, en_key, fr_key=None, default=None):
    """Récupère une valeur YAML avec priorité anglais, fallback français"""
    if en_key in obj:
        return obj[en_key]
    if fr_key and fr_key in obj:
        return obj[fr_key]
    return default

# Texte d'aide pour les champs disponibles
HELP_TEXT = """
YAML STRUCTURE (English)
════════════════════════
title: "Graph title"
type: line|bar|stacked_bar
x_axis:
  field: ANNE
  label: "X axis"
y_axis:
  field: PIBS
  label: "Y axis"
  divisor: 1
  min: 0
curves:
  - field: PIBS
    label: "Label"
    color: "#93c5fd"
    style: solid|dashed|dotted
    fill: true|false
    fill_alpha: 0.2

FIELD CODES (4 letters)
═══════════════════════
ANNE  annee
      Year of simulation

PIBS  pib
      GDP in Bn

DIFF  differentiel_pct
      Differential (% GDP)

INTR  interets_totaux
      Interest (Bn/year)

DPUB  dette_publique
      Public debt (Bn)

DTRA  dette_transition
      Transition debt (Bn)

DTOT  dette_publique_totale
      Total debt (Bn)

DPPC  dette_publique_pct
      Public debt (% GDP)

DTPC  dette_transition_pct
      Transition debt (% GDP)

TTPC  dette_publique_totale_pct
      Total debt (% GDP)

DIMP  dette_implicite
      Implicit debt (Bn)

DIPC  dette_implicite_pct
      Implicit debt (% GDP)

FLUX  flux_pensions
      Pensions (Bn/year)

NRET  nombre_retraites
      Number of retirees

REQR  retraites_equiv_repartition
      Equiv. retirees (weighted by rights)

PARAM CODES (4 letters)
═══════════════════════
PIBI  pib_initial
      Initial GDP (Bn)

DPBI  dette_publique_initiale
      Initial public debt (Bn)

TCBS  taux_croissance_base
      Base growth rate (%)

DIBI  dette_implicite_initiale
      Initial implicit debt (Bn)

RSOL  reduction_solidaire
      Solidarity reduction (%)

PMOY  pension_moyenne_annuelle
      Average annual pension

PRIV  privatisations_totales
      Total privatizations (Bn)

DFIN  differentiel_initial
      Initial differential (%)

DDEC  duree_decroissance_diff
      Differential decrease (yrs)

TFLT  taux_flat_tax
      Flat tax rate (%)

ABAT  abattement_forfaitaire
      Flat deduction (/month)

ASAN  assurance_sante
      Health insurance

ACHO  assurance_chomage
      Unemployment insurance

APEN  assurance_pension
      Pension insurance

AEDU  assurance_education
      Education insurance

VARIABLES
─────────
{devise} = Currency symbol

SAVE GRAPH
──────────
💾 = Save to SVG/PNG
Filename format:
scenario_graph_CODE=val
"""


def load_graph_files():
    """Charge tous les fichiers YAML de graphiques depuis le dossier graphiques/"""
    graphs = {}
    if GRAPHIQUES_DIR.exists():
        for yaml_file in sorted(GRAPHIQUES_DIR.glob("*.yaml")):
            if yaml_file.name.startswith('_'):
                continue
            try:
                with open(yaml_file, 'r', encoding='utf-8') as f:
                    data = yaml.safe_load(f)
                    if data and isinstance(data, dict):
                        graph_id = yaml_file.stem
                        data['_file'] = yaml_file
                        graphs[graph_id] = data
            except Exception as e:
                print(f"Erreur chargement {yaml_file}: {e}")
    return graphs


def count_config_files():
    """Compte le nombre de fichiers de configuration disponibles"""
    if CONFIG_DIR.exists():
        return len(list(CONFIG_DIR.rglob('*.ini')))
    return 0


class GraphPanel(ttk.Frame):
    """Panneau contenant un graphique avec sélecteur et éditeur intégré"""

    def __init__(self, parent, on_click_enlarge, on_edit_graph, panel_index, font_size=BASE_FONT_SIZE, on_graph_reload=None, get_modified_params_callback=None, get_scenario_name_callback=None, on_new_graph=None, on_expand_inplace=None, on_status_update=None, read_only=False):
        super().__init__(parent)
        self.on_click_enlarge = on_click_enlarge
        self.on_edit_graph = on_edit_graph
        self.on_graph_reload = on_graph_reload
        self.get_modified_params_callback = get_modified_params_callback
        self.get_scenario_name_callback = get_scenario_name_callback
        self.on_new_graph = on_new_graph
        self.on_expand_inplace = on_expand_inplace
        self.on_status_update = on_status_update
        self.panel_index = panel_index
        self.read_only = read_only  # Mode lecture seule (pour comparaison)
        self.current_graph_id = None
        self.current_graph_def = None
        self.resultats = None
        self.params = None
        self.devise = "€"
        self.graph_files = {}
        self.font_size = font_size
        self.edit_mode = False
        self.text_edit_mode = False  # True = éditeur texte, False = éditeur visuel
        self.current_file_path = None
        self.curve_visibility = {}  # {curve_label: BooleanVar} pour visibilité des courbes
        self.yaml_editable = False  # Verrouillage éditeur YAML

        self._create_widgets()

    def _update_status(self, message, error=False):
        """Met à jour la barre de statut via le callback"""
        if self.on_status_update:
            self.on_status_update(message, error=error)

    def _create_widgets(self):
        # === BARRE D'OUTILS UNIQUE (change selon le mode) ===
        self.toolbar_frame = ttk.Frame(self)
        # En mode read_only (comparaison), barre plus compacte
        if not self.read_only:
            self.toolbar_frame.pack(fill='x', padx=5, pady=2)

            # --- Zone gauche : sélection graphique ---
            self.graph_label = ttk.Label(self.toolbar_frame, text=t('panel.graph_select'))
            self.graph_label.pack(side='left')

            self.graph_var = tk.StringVar()
            self.graph_combo = ttk.Combobox(
                self.toolbar_frame,
                textvariable=self.graph_var,
                state='readonly',
                width=25
            )
            self.graph_combo.pack(side='left', padx=5)
            self.graph_combo.bind('<<ComboboxSelected>>', self._on_graph_selected)

            # --- Boutons MODE GRAPHIQUE (visibles par défaut) ---
            self.graph_buttons_frame = ttk.Frame(self.toolbar_frame)
            self.graph_buttons_frame.pack(side='left', padx=2)

            self.edit_btn = ttk.Button(self.graph_buttons_frame, text="✎", width=3, command=self._toggle_edit_mode)
            self.edit_btn.pack(side='left', padx=1)
            self._add_tooltip(self.edit_btn, t('tooltip.edit_yaml'))

            self.save_btn = ttk.Button(self.graph_buttons_frame, text="💾", width=3, command=self._save_graph)
            self.save_btn.pack(side='left', padx=1)
            self._add_tooltip(self.save_btn, t('tooltip.save_graph'))

            self.copy_png_btn = ttk.Button(self.graph_buttons_frame, text="📋", width=3, command=self._copy_graph_png)
            self.copy_png_btn.pack(side='left', padx=1)
            self._add_tooltip(self.copy_png_btn, t('tooltip.copy_png'))

            self.copy_svg_btn = ttk.Button(self.graph_buttons_frame, text="📄", width=3, command=self._copy_graph_svg)
            self.copy_svg_btn.pack(side='left', padx=1)
            self._add_tooltip(self.copy_svg_btn, t('tooltip.copy_svg'))

            self.reset_zoom_btn = ttk.Button(self.graph_buttons_frame, text="⟲", width=3, command=self._reset_zoom)
            self.reset_zoom_btn.pack(side='left', padx=1)
            self._add_tooltip(self.reset_zoom_btn, t('tooltip.reset_zoom'))

            # --- Boutons MODE ÉDITION (cachés par défaut) ---
            self.edit_buttons_frame = ttk.Frame(self.toolbar_frame)
            # Ne pas pack - sera affiché quand on entre en mode édition

            self.save_edit_btn = ttk.Button(self.edit_buttons_frame, text="💾", width=3, command=self._save_current_editor)
            self.save_edit_btn.pack(side='left', padx=1)
            self._add_tooltip(self.save_edit_btn, t('tooltip.save'))

            self.validate_edit_btn = ttk.Button(self.edit_buttons_frame, text="✓", width=3, command=self._validate_current_editor)
            self.validate_edit_btn.pack(side='left', padx=1)
            self._add_tooltip(self.validate_edit_btn, t('tooltip.validate'))

            self.switch_mode_btn = ttk.Button(self.edit_buttons_frame, text="🎨", width=3, command=self._switch_editor_mode)
            self.switch_mode_btn.pack(side='left', padx=1)
            self._add_tooltip(self.switch_mode_btn, t('tooltip.switch_mode'))

            # Label nom fichier
            self.file_label = ttk.Label(self.edit_buttons_frame, text="")
            self.file_label.pack(side='left', padx=5)

            # Cadenas pour verrouillage
            self.yaml_edit_var = tk.BooleanVar(value=False)
            self.yaml_lock_check = ttk.Checkbutton(
                self.edit_buttons_frame,
                text="🔒",
                variable=self.yaml_edit_var,
                command=self._toggle_yaml_edit_mode
            )
            self.yaml_lock_check.pack(side='left', padx=2)
            self._add_tooltip(self.yaml_lock_check, t('tooltip.lock'))

            self.close_edit_btn = ttk.Button(self.edit_buttons_frame, text="✕", width=3, command=self._close_current_editor)
            self.close_edit_btn.pack(side='left', padx=1)
            self._add_tooltip(self.close_edit_btn, t('tooltip.close'))

            # Status pour l'édition
            self.edit_status_var = tk.StringVar(value="")
            ttk.Label(self.edit_buttons_frame, textvariable=self.edit_status_var).pack(side='right', padx=5)

            # --- Boutons de droite (toujours visibles) ---
            self.enlarge_btn = ttk.Button(self.toolbar_frame, text="⤢", width=3, command=self._on_enlarge_click)
            self.enlarge_btn.pack(side='right')
            self._add_tooltip(self.enlarge_btn, t('tooltip.enlarge'))

            self.expand_btn = ttk.Button(self.toolbar_frame, text="⬚", width=3, command=self._on_expand_inplace_click)
            self.expand_btn.pack(side='right', padx=2)
            self._add_tooltip(self.expand_btn, t('tooltip.expand'))
        else:
            # Mode read_only : pas de toolbar, juste le graphique
            self.graph_var = tk.StringVar()
            self.graph_combo = None
            self.graph_buttons_frame = None
            self.edit_buttons_frame = None
            self.edit_status_var = tk.StringVar(value="")
            self.yaml_edit_var = tk.BooleanVar(value=False)

        # === PANNEAU VISIBILITÉ DES COURBES (caché par défaut) ===
        self.curves_frame = ttk.Frame(self)
        # Ne pas pack maintenant - sera affiché quand il y a plusieurs courbes
        self.curve_checkboxes = []

        # === CONTENEUR PRINCIPAL (graphique OU éditeur) ===
        self.content_frame = ttk.Frame(self)
        self.content_frame.pack(fill='both', expand=True)
        self.content_frame.pack_propagate(False)  # Empêcher le redimensionnement lors du changement de mode

        # --- GRAPHIQUE ---
        self.graph_frame = ttk.Frame(self.content_frame)
        self.graph_frame.pack(fill='both', expand=True)

        # Figure matplotlib (pour générer l'image)
        self.fig = Figure(figsize=(5, 4), dpi=100)
        self.ax = self.fig.add_subplot(111)

        # Canvas tkinter pour afficher l'image avec zoom/pan
        self.display_canvas = tk.Canvas(self.graph_frame, bg='white', highlightthickness=0)
        self.display_canvas.pack(fill='both', expand=True, padx=5, pady=5)

        # Image affichée
        self._graph_image = None
        self._photo_image = None
        self._canvas_image_id = None

        # Zoom et pan état
        self._zoom_level = 1.0
        self._pan_offset_x = 0
        self._pan_offset_y = 0
        self._drag_start_x = 0
        self._drag_start_y = 0

        # Bind pour zoom avec molette (Ctrl=graphique, sans Ctrl=image)
        self.display_canvas.bind('<MouseWheel>', self._on_mousewheel)
        self.display_canvas.bind('<Button-4>', self._on_scroll_up)   # Linux scroll up
        self.display_canvas.bind('<Button-5>', self._on_scroll_down) # Linux scroll down

        # Bind pour pan avec drag (bouton gauche)
        self.display_canvas.bind('<ButtonPress-1>', self._on_drag_start)
        self.display_canvas.bind('<B1-Motion>', self._on_drag_motion)

        # Double-clic gauche pour agrandir
        self.display_canvas.bind('<Double-Button-1>', lambda e: self._on_enlarge_click())

        # Double-clic droit pour reset zoom/pan
        self.display_canvas.bind('<Double-Button-3>', lambda e: self._reset_zoom())

        # Redimensionnement auto
        self.display_canvas.bind('<Configure>', self._on_canvas_resize)

        # --- ÉDITEUR VISUEL (caché par défaut) ---
        self.visual_editor_frame = ttk.Frame(self.content_frame)
        self._create_visual_editor_widgets()

        # --- ÉDITEUR TEXTE (caché par défaut) ---
        self.editor_frame = ttk.Frame(self.content_frame)
        # Ne pas pack maintenant - sera affiché quand on bascule en mode édition
        # La barre d'outils est maintenant unifiée (edit_buttons_frame dans toolbar_frame)

        # Zone de texte éditeur
        self.editor_text = tk.Text(self.editor_frame, wrap='none', font=('Consolas', self.font_size))
        editor_scroll_y = ttk.Scrollbar(self.editor_frame, orient='vertical', command=self.editor_text.yview)
        editor_scroll_x = ttk.Scrollbar(self.editor_frame, orient='horizontal', command=self.editor_text.xview)
        self.editor_text.configure(yscrollcommand=editor_scroll_y.set, xscrollcommand=editor_scroll_x.set)

        editor_scroll_y.pack(side='right', fill='y')
        editor_scroll_x.pack(side='bottom', fill='x')
        self.editor_text.pack(fill='both', expand=True, padx=2, pady=2)

        # Binding Select All (Ctrl+A)
        self.editor_text.bind('<Control-a>', lambda e: self._select_all_text(e.widget))
        self.editor_text.bind('<Control-A>', lambda e: self._select_all_text(e.widget))

        # Initialiser l'état du verrouillage YAML
        self._update_yaml_editor_state()

    def _select_all_text(self, widget):
        """Sélectionne tout le texte dans un widget Text"""
        widget.tag_add('sel', '1.0', 'end')
        widget.mark_set('insert', 'end')
        return 'break'  # Empêche le comportement par défaut

    def _add_tooltip(self, widget, text):
        """Ajoute un tooltip à un widget"""
        def show_tooltip(event):
            tooltip = tk.Toplevel(widget)
            tooltip.wm_overrideredirect(True)
            tooltip.wm_geometry(f"+{event.x_root+10}+{event.y_root+10}")
            label = tk.Label(tooltip, text=text, background="#ffffe0", relief='solid', borderwidth=1,
                           font=('TkDefaultFont', self.font_size - 2))
            label.pack()
            widget._tooltip = tooltip
            widget.after(2000, lambda: tooltip.destroy() if hasattr(widget, '_tooltip') else None)

        def hide_tooltip(event):
            if hasattr(widget, '_tooltip') and widget._tooltip:
                widget._tooltip.destroy()
                widget._tooltip = None

        widget.bind('<Enter>', show_tooltip)
        widget.bind('<Leave>', hide_tooltip)

    def _show_toolbar_mode(self, mode):
        """Affiche les boutons appropriés selon le mode (graph/edit)"""
        if mode == 'graph':
            self.edit_buttons_frame.pack_forget()
            self.graph_buttons_frame.pack(side='left', padx=2)
        elif mode == 'edit':
            self.graph_buttons_frame.pack_forget()
            self.edit_buttons_frame.pack(side='left', padx=2)

    def _save_current_editor(self):
        """Sauvegarde selon le mode d'édition actuel"""
        if self.text_edit_mode:
            self._save_editor()
        else:
            self._save_visual_editor()

    def _validate_current_editor(self):
        """Valide selon le mode d'édition actuel"""
        if self.text_edit_mode:
            self._validate_editor()
        else:
            # Pour le mode visuel, pas de validation séparée
            pass

    def _switch_editor_mode(self):
        """Bascule entre mode texte et mode visuel"""
        if self.text_edit_mode:
            self._switch_to_visual_mode()
        else:
            self._switch_to_text_mode()

    def _close_current_editor(self):
        """Ferme l'éditeur actuel"""
        if self.text_edit_mode:
            self._close_editor()
        else:
            self._close_visual_editor()

    def _toggle_yaml_edit_mode(self):
        """Bascule le verrouillage de l'éditeur YAML"""
        self.yaml_editable = self.yaml_edit_var.get()
        self._update_yaml_editor_state()

    def _update_yaml_editor_state(self):
        """Met à jour l'état de l'éditeur selon le verrouillage (texte et visuel)"""
        # Mettre à jour l'icône du cadenas
        lock_icon = "🔓" if self.yaml_editable else "🔒"
        self.yaml_lock_check.configure(text=lock_icon)

        # Mettre à jour l'état et la couleur de l'éditeur texte
        if self.yaml_editable:
            self.editor_text.configure(
                state='normal',
                background='white',
                foreground='black'
            )
        else:
            self.editor_text.configure(
                state='disabled',
                background=COLORS['readonly_bg'],
                foreground='#666666'
            )

        # Mettre à jour l'état des boutons de la barre unifiée
        state = 'normal' if self.yaml_editable else 'disabled'
        self.save_edit_btn.configure(state=state)
        self.validate_edit_btn.configure(state=state)

        # Mettre à jour l'état des widgets de l'éditeur visuel
        self._update_visual_editor_state()

    def _update_visual_editor_state(self):
        """Met à jour l'état des widgets de l'éditeur visuel selon le verrouillage"""
        state = 'normal' if self.yaml_editable else 'disabled'
        readonly_state = 'readonly' if self.yaml_editable else 'disabled'

        # Widgets généraux
        if hasattr(self, 'visual_type_combo'):
            self.visual_type_combo.configure(state=readonly_state)
        if hasattr(self, 'visual_x_field_combo'):
            self.visual_x_field_combo.configure(state=readonly_state)

        # Parcourir les widgets dans visual_scrollable_frame pour mettre à jour leur état
        if hasattr(self, 'visual_scrollable_frame'):
            for child in self.visual_scrollable_frame.winfo_children():
                self._set_widget_state_recursive(child, state, readonly_state)

    def _set_widget_state_recursive(self, widget, state, readonly_state):
        """Met à jour l'état d'un widget et de ses enfants récursivement"""
        widget_class = widget.winfo_class()

        if widget_class == 'TEntry':
            widget.configure(state=state)
        elif widget_class == 'TCombobox':
            widget.configure(state=readonly_state)
        elif widget_class == 'TCheckbutton':
            widget.configure(state=state)
        elif widget_class == 'TButton':
            widget.configure(state=state)
        elif widget_class == 'Button':
            widget.configure(state=state)

        # Parcourir les enfants
        for child in widget.winfo_children():
            self._set_widget_state_recursive(child, state, readonly_state)

    def _toggle_edit_mode(self):
        """Bascule entre le graphique et l'éditeur visuel"""
        if self.edit_mode:
            self._close_visual_editor()
        else:
            self._open_visual_editor()

    def _open_editor(self):
        """Ouvre l'éditeur texte intégré à la place du graphique"""
        if not self.current_graph_id:
            return

        graph_def = self.graph_files.get(self.current_graph_id)
        if not graph_def or '_file' not in graph_def:
            return

        self.current_file_path = graph_def['_file']

        # Charger le contenu du fichier
        try:
            # Mettre à jour l'état de l'éditeur AVANT de charger
            self._update_yaml_editor_state()

            # Activer temporairement pour charger le contenu
            self.editor_text.configure(state='normal')
            content = self.current_file_path.read_text(encoding='utf-8')
            self.editor_text.delete('1.0', 'end')
            self.editor_text.insert('1.0', content)

            # Remettre l'état correct selon le verrouillage
            if not self.yaml_editable:
                self.editor_text.configure(
                    state='disabled',
                    background=COLORS['readonly_bg'],
                    foreground='#666666'
                )

            # Mettre à jour le nom du fichier et le status
            self.file_label.configure(text=f"📝 {self.current_file_path.name}")
            self.edit_status_var.set("")
        except Exception as e:
            self.edit_status_var.set(f"Erreur: {e}")
            return

        # Basculer la barre d'outils et l'affichage
        self._show_toolbar_mode('edit')
        self.switch_mode_btn.configure(text="🎨")  # Mode visuel disponible
        self.graph_frame.pack_forget()
        self.editor_frame.pack(fill='both', expand=True)
        self.edit_mode = True
        self.text_edit_mode = True

    def _close_editor(self):
        """Ferme l'éditeur texte et retourne au graphique"""
        self.editor_frame.pack_forget()
        self.visual_editor_frame.pack_forget()
        self.graph_frame.pack(fill='both', expand=True)
        self.edit_mode = False
        self.text_edit_mode = False
        self._show_toolbar_mode('graph')
        self.edit_status_var.set("")
        self.file_label.configure(text="")

    def _save_editor(self):
        """Sauvegarde le contenu de l'éditeur"""
        if not self.current_file_path:
            return

        content = self.editor_text.get('1.0', 'end-1c')

        # Valider la syntaxe YAML
        try:
            yaml.safe_load(content)
        except Exception as e:
            result = messagebox.askyesno(t('dialogs.error'),
                t('dialogs.yaml_syntax_error', error=e))
            if not result:
                return

        try:
            self.current_file_path.write_text(content, encoding='utf-8')
            self.edit_status_var.set(f"✓ {t('editor.saved')}")

            # Recharger les graphiques si callback fourni
            if self.on_graph_reload:
                self.on_graph_reload()
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_save', error=e))

    def _validate_editor(self):
        """Valide la syntaxe YAML"""
        content = self.editor_text.get('1.0', 'end-1c')
        try:
            yaml.safe_load(content)
            self.edit_status_var.set(f"✓ {t('editor.syntax_ok')}")
        except Exception as e:
            self.edit_status_var.set(f"✗ {t('editor.error')}")
            messagebox.showerror(t('dialogs.error'), str(e))

    # ============================================================================
    # === ÉDITEUR VISUEL (intégré dans le panneau)
    # ============================================================================

    # Constants for visual editor
    FIELD_CODES = ['ANNE', 'PIBS', 'DIFF', 'INTR', 'DPUB', 'DTRA', 'DTOT', 'DPPC', 'DTPC', 'TTPC', 'DIMP', 'DIPC', 'FLUX', 'NRET']
    GRAPH_TYPE_CODES = ['line', 'bar', 'stacked_bar']
    LINE_STYLE_CODES = ['solid', 'dashed', 'dotted']
    COLORS_PRESETS = [
        '#4A90E2', '#93c5fd', '#16a34a', '#86efac', '#f97316', '#fdba74',
        '#ef4444', '#fca5a5', '#8b5cf6', '#c4b5fd', '#374151', '#d1d5db',
    ]

    def _get_field_options(self):
        """Get field options with translated labels"""
        return [(code, t(f'fields.{code}')) for code in self.FIELD_CODES]

    def _get_graph_types(self):
        """Get graph types with translated labels"""
        return [(code, t(f'graph_types.{code}')) for code in self.GRAPH_TYPE_CODES]

    def _get_line_styles(self):
        """Get line styles with translated labels"""
        return [(code, t(f'line_styles.{code}')) for code in self.LINE_STYLE_CODES]

    def _create_visual_editor_widgets(self):
        """Crée les widgets de l'éditeur visuel (intégré dans le panneau)"""
        self.visual_curve_widgets = []
        # La barre d'outils est maintenant unifiée (edit_buttons_frame dans toolbar_frame)
        self.visual_status_var = tk.StringVar(value="")

        # === CONTENU SCROLLABLE ===
        visual_canvas = tk.Canvas(self.visual_editor_frame)
        visual_scrollbar = ttk.Scrollbar(self.visual_editor_frame, orient="vertical", command=visual_canvas.yview)
        self.visual_scrollable_frame = ttk.Frame(visual_canvas)

        self.visual_scrollable_frame.bind(
            "<Configure>",
            lambda e: visual_canvas.configure(scrollregion=visual_canvas.bbox("all"))
        )

        visual_canvas.create_window((0, 0), window=self.visual_scrollable_frame, anchor="nw")
        visual_canvas.configure(yscrollcommand=visual_scrollbar.set)

        visual_scrollbar.pack(side="right", fill="y")
        visual_canvas.pack(side="left", fill="both", expand=True)

        # Bind mouse wheel for scrolling
        def _on_mousewheel(event):
            visual_canvas.yview_scroll(int(-1*(event.delta/120)), "units")
        visual_canvas.bind_all("<MouseWheel>", _on_mousewheel)

        # === PROPRIÉTÉS GÉNÉRALES ===
        general_frame = ttk.LabelFrame(self.visual_scrollable_frame, text=f"📊 {t('editor.general_props')}")
        general_frame.pack(fill='x', padx=10, pady=5)

        # Titre
        ttk.Label(general_frame, text=t('editor.graph_title')).grid(row=0, column=0, sticky='w', padx=5, pady=3)
        self.visual_title_var = tk.StringVar()
        ttk.Entry(general_frame, textvariable=self.visual_title_var, width=50).grid(row=0, column=1, columnspan=2, sticky='w', padx=5, pady=3)

        # Type de graphique
        ttk.Label(general_frame, text=t('editor.graph_type')).grid(row=1, column=0, sticky='w', padx=5, pady=3)
        self.visual_type_var = tk.StringVar()
        self.visual_type_combo = ttk.Combobox(general_frame, textvariable=self.visual_type_var, state='readonly', width=20)
        self.visual_type_combo['values'] = [f"{code} - {name}" for code, name in self._get_graph_types()]
        self.visual_type_combo.grid(row=1, column=1, sticky='w', padx=5, pady=3)

        # === AXE X ===
        x_frame = ttk.LabelFrame(self.visual_scrollable_frame, text=f"↔️ {t('editor.x_axis')}")
        x_frame.pack(fill='x', padx=10, pady=5)

        ttk.Label(x_frame, text=t('editor.field')).grid(row=0, column=0, sticky='w', padx=5, pady=3)
        self.visual_x_field_var = tk.StringVar()
        self.visual_x_field_combo = ttk.Combobox(x_frame, textvariable=self.visual_x_field_var, state='readonly', width=30)
        self.visual_x_field_combo['values'] = [f"{code} - {name}" for code, name in self._get_field_options()]
        self.visual_x_field_combo.grid(row=0, column=1, sticky='w', padx=5, pady=3)

        ttk.Label(x_frame, text=t('editor.label')).grid(row=1, column=0, sticky='w', padx=5, pady=3)
        self.visual_x_label_var = tk.StringVar()
        ttk.Entry(x_frame, textvariable=self.visual_x_label_var, width=40).grid(row=1, column=1, sticky='w', padx=5, pady=3)

        # === AXE Y ===
        y_frame = ttk.LabelFrame(self.visual_scrollable_frame, text=f"↕️ {t('editor.y_axis')}")
        y_frame.pack(fill='x', padx=10, pady=5)

        ttk.Label(y_frame, text=t('editor.label')).grid(row=0, column=0, sticky='w', padx=5, pady=3)
        self.visual_y_label_var = tk.StringVar()
        ttk.Entry(y_frame, textvariable=self.visual_y_label_var, width=40).grid(row=0, column=1, sticky='w', padx=5, pady=3)

        ttk.Label(y_frame, text=t('editor.divisor')).grid(row=1, column=0, sticky='w', padx=5, pady=3)
        self.visual_y_divisor_var = tk.StringVar(value="1")
        ttk.Entry(y_frame, textvariable=self.visual_y_divisor_var, width=10).grid(row=1, column=1, sticky='w', padx=5, pady=3)

        ttk.Label(y_frame, text=t('editor.min')).grid(row=2, column=0, sticky='w', padx=5, pady=3)
        self.visual_y_min_var = tk.StringVar()
        ttk.Entry(y_frame, textvariable=self.visual_y_min_var, width=10).grid(row=2, column=1, sticky='w', padx=5, pady=3)

        # === COURBES ===
        self.visual_curves_frame = ttk.LabelFrame(self.visual_scrollable_frame, text=f"📈 {t('editor.curves')}")
        self.visual_curves_frame.pack(fill='x', padx=10, pady=5)

        ttk.Button(self.visual_curves_frame, text=f"➕ {t('editor.add_curve')}", command=self._add_visual_curve_widget).pack(pady=5)

        self.visual_curves_container = ttk.Frame(self.visual_curves_frame)
        self.visual_curves_container.pack(fill='x', padx=5, pady=5)

    def _open_visual_editor(self):
        """Ouvre l'éditeur visuel à la place du graphique"""
        if not self.current_graph_id:
            return

        graph_def = self.graph_files.get(self.current_graph_id)
        if not graph_def or '_file' not in graph_def:
            return

        self.current_file_path = graph_def['_file']
        self._load_visual_editor()

        # Mettre à jour l'état de verrouillage avant d'afficher
        self._update_yaml_editor_state()

        # Basculer la barre d'outils et l'affichage
        self._show_toolbar_mode('edit')
        self.switch_mode_btn.configure(text="📝")  # Mode texte disponible
        self.file_label.configure(text=f"🎨 {self.current_file_path.name}")
        self.edit_status_var.set("")
        self.graph_frame.pack_forget()
        self.editor_frame.pack_forget()
        self.visual_editor_frame.pack(fill='both', expand=True)
        self.edit_mode = True
        self.text_edit_mode = False
        self.edit_btn.configure(text="📊")

    def _close_visual_editor(self):
        """Ferme l'éditeur visuel et retourne au graphique"""
        self.visual_editor_frame.pack_forget()
        self.editor_frame.pack_forget()
        self.graph_frame.pack(fill='both', expand=True)
        self.edit_mode = False
        self.text_edit_mode = False
        self._show_toolbar_mode('graph')
        self.edit_btn.configure(text="✎")
        self.edit_status_var.set("")
        self.file_label.configure(text="")

    def _switch_to_visual_mode(self):
        """Bascule de l'éditeur texte vers l'éditeur visuel"""
        # D'abord sauvegarder le texte si modifié (seulement si on est en mode texte et déverrouillé)
        if self.text_edit_mode and self.yaml_editable and self.current_file_path:
            content = self.editor_text.get('1.0', 'end-1c')
            if content.strip():  # Ne pas sauvegarder si vide
                try:
                    yaml.safe_load(content)  # Valider
                    self.current_file_path.write_text(content, encoding='utf-8')
                except Exception:
                    pass  # Ignorer les erreurs de syntaxe

        # Charger dans l'éditeur visuel
        self._load_visual_editor()

        # Mettre à jour l'état de verrouillage
        self._update_visual_editor_state()

        # Mettre à jour la barre d'outils
        self.switch_mode_btn.configure(text="📝")  # Mode texte disponible
        self.file_label.configure(text=f"🎨 {self.current_file_path.name}")
        self.text_edit_mode = False

        # Basculer l'affichage
        self.editor_frame.pack_forget()
        self.visual_editor_frame.pack(fill='both', expand=True)

    def _switch_to_text_mode(self):
        """Bascule de l'éditeur visuel vers l'éditeur texte"""
        # D'abord sauvegarder les données visuelles (seulement si on est en mode visuel)
        if not self.text_edit_mode:
            self._save_visual_editor_to_file()

        # Mettre à jour l'état de l'éditeur AVANT de charger
        self._update_yaml_editor_state()

        # Recharger le texte
        if self.current_file_path:
            try:
                # Activer temporairement pour charger le contenu
                self.editor_text.configure(state='normal')
                content = self.current_file_path.read_text(encoding='utf-8')
                self.editor_text.delete('1.0', 'end')
                self.editor_text.insert('1.0', content)

                # Remettre l'état correct selon le verrouillage
                if not self.yaml_editable:
                    self.editor_text.configure(
                        state='disabled',
                        background=COLORS['readonly_bg'],
                        foreground='#666666'
                    )
            except Exception as e:
                self.edit_status_var.set(f"Erreur: {e}")

        # Mettre à jour la barre d'outils
        self.switch_mode_btn.configure(text="🎨")  # Mode visuel disponible
        self.file_label.configure(text=f"📝 {self.current_file_path.name}")
        self.text_edit_mode = True

        # Basculer l'affichage
        self.visual_editor_frame.pack_forget()
        self.editor_frame.pack(fill='both', expand=True)

    def _load_visual_editor(self):
        """Charge le fichier YAML et remplit les widgets visuels"""
        if not self.current_file_path:
            return

        try:
            content = self.current_file_path.read_text(encoding='utf-8')
            graph_data = yaml.safe_load(content) or {}
            self._populate_visual_widgets(graph_data)
            self.visual_status_var.set(f"✓ {t('editor.file_loaded')}")
        except Exception as e:
            self.visual_status_var.set(f"Erreur: {e}")

    def _populate_visual_widgets(self, data):
        """Remplit les widgets visuels avec les données du YAML"""
        # Titre
        self.visual_title_var.set(get_yaml_value(data, 'title', 'titre', ''))

        # Type
        graph_type = data.get('type', 'line')
        for code, name in self._get_graph_types():
            if code == graph_type:
                self.visual_type_var.set(f"{code} - {name}")
                break

        # Axe X
        x_axis = get_yaml_value(data, 'x_axis', 'axe_x', {})
        x_field = get_yaml_value(x_axis, 'field', 'champ', '')
        for code, name in self._get_field_options():
            if code == x_field:
                self.visual_x_field_var.set(f"{code} - {name}")
                break
        self.visual_x_label_var.set(get_yaml_value(x_axis, 'label', 'label', ''))

        # Axe Y
        y_axis = get_yaml_value(data, 'y_axis', 'axe_y', {})
        self.visual_y_label_var.set(get_yaml_value(y_axis, 'label', 'label', ''))
        self.visual_y_divisor_var.set(str(get_yaml_value(y_axis, 'divisor', 'diviseur', 1)))
        y_min = y_axis.get('min')
        self.visual_y_min_var.set(str(y_min) if y_min is not None else '')

        # Supprimer les anciennes courbes
        for w in self.visual_curve_widgets:
            if 'frame' in w:
                w['frame'].destroy()
        self.visual_curve_widgets = []

        # Courbes
        curves = get_yaml_value(data, 'curves', 'courbes', [])
        for curve in curves:
            field_code = get_yaml_value(curve, 'field', 'champ', '')
            # Trouver le bon format pour le combobox
            for code, name in self._get_field_options():
                if code == field_code:
                    curve['field'] = f"{code} - {name}"
                    break
            self._add_visual_curve_widget(curve)

    def _add_visual_curve_widget(self, data=None):
        """Ajoute un widget pour une courbe dans l'éditeur visuel"""
        idx = len(self.visual_curve_widgets)
        frame = ttk.LabelFrame(self.visual_curves_container, text=t('editor.curve_n', n=idx + 1))
        frame.pack(fill='x', pady=3)

        widgets = {}

        # Ligne 1: Champ + Label
        ttk.Label(frame, text=t('editor.field')).grid(row=0, column=0, sticky='w', padx=3, pady=2)
        widgets['field'] = tk.StringVar(value=data.get('field', '') if data else '')
        field_combo = ttk.Combobox(frame, textvariable=widgets['field'], state='readonly', width=25)
        field_combo['values'] = [f"{code} - {name}" for code, name in self._get_field_options()]
        field_combo.grid(row=0, column=1, sticky='w', padx=3, pady=2)

        ttk.Label(frame, text=t('editor.label')).grid(row=0, column=2, sticky='w', padx=3, pady=2)
        widgets['label'] = tk.StringVar(value=data.get('label', '') if data else '')
        ttk.Entry(frame, textvariable=widgets['label'], width=20).grid(row=0, column=3, sticky='w', padx=3, pady=2)

        # Ligne 2: Couleur + Style
        ttk.Label(frame, text=t('editor.color')).grid(row=1, column=0, sticky='w', padx=3, pady=2)
        widgets['color'] = tk.StringVar(value=data.get('color', '#4A90E2') if data else '#4A90E2')

        color_frame = ttk.Frame(frame)
        color_frame.grid(row=1, column=1, sticky='w', padx=3, pady=2)

        color_combo = ttk.Combobox(color_frame, textvariable=widgets['color'], width=15)
        color_combo['values'] = self.COLORS_PRESETS
        color_combo.pack(side='left')

        # Bouton sélecteur de couleur
        widgets['color_btn'] = tk.Button(color_frame, text="🎨", width=3,
                                         command=lambda: self._pick_visual_color(widgets['color']))
        widgets['color_btn'].pack(side='left', padx=2)

        ttk.Label(frame, text=t('editor.style')).grid(row=1, column=2, sticky='w', padx=3, pady=2)
        widgets['style'] = tk.StringVar(value=data.get('style', 'solid') if data else 'solid')
        style_combo = ttk.Combobox(frame, textvariable=widgets['style'], state='readonly', width=15)
        style_combo['values'] = [code for code, name in self._get_line_styles()]
        style_combo.grid(row=1, column=3, sticky='w', padx=3, pady=2)

        # Ligne 3: Fill + Alpha
        widgets['fill'] = tk.BooleanVar(value=data.get('fill', False) if data else False)
        ttk.Checkbutton(frame, text=t('editor.fill'), variable=widgets['fill']).grid(row=2, column=0, columnspan=2, sticky='w', padx=3, pady=2)

        ttk.Label(frame, text=t('editor.alpha')).grid(row=2, column=2, sticky='w', padx=3, pady=2)
        widgets['fill_alpha'] = tk.StringVar(value=str(data.get('fill_alpha', 0.2)) if data else '0.2')
        ttk.Entry(frame, textvariable=widgets['fill_alpha'], width=8).grid(row=2, column=3, sticky='w', padx=3, pady=2)

        # Bouton supprimer
        ttk.Button(frame, text="🗑️", width=3,
                   command=lambda f=frame: self._remove_visual_curve(f)).grid(row=0, column=4, rowspan=3, padx=5)

        widgets['frame'] = frame
        self.visual_curve_widgets.append(widgets)

    def _remove_visual_curve(self, frame):
        """Supprime une courbe de l'éditeur visuel"""
        frame.destroy()
        self.visual_curve_widgets = [w for w in self.visual_curve_widgets if w.get('frame') != frame]
        # Renuméroter les courbes
        for i, w in enumerate(self.visual_curve_widgets):
            w['frame'].configure(text=t('editor.curve_n', n=i + 1))

    def _pick_visual_color(self, color_var):
        """Ouvre le sélecteur de couleur pour l'éditeur visuel"""
        from tkinter import colorchooser
        color = colorchooser.askcolor(color=color_var.get(), title="Choisir une couleur")
        if color[1]:
            color_var.set(color[1])

    def _save_visual_editor(self):
        """Sauvegarde les données de l'éditeur visuel et recharge le graphique"""
        self._save_visual_editor_to_file()
        # Recharger les graphiques si callback fourni
        if self.on_graph_reload:
            self.on_graph_reload()

    def _save_visual_editor_to_file(self):
        """Sauvegarde les données de l'éditeur visuel dans le fichier YAML"""
        if not self.current_file_path:
            return

        # Protection contre la sauvegarde de données vides
        if not self.visual_title_var.get() and not self.visual_type_var.get():
            # Ne pas sauvegarder si les données semblent vides
            return

        try:
            # Construire le dictionnaire YAML
            data = {
                'title': self.visual_title_var.get(),
                'type': self.visual_type_var.get().split(' - ')[0] if ' - ' in self.visual_type_var.get() else self.visual_type_var.get(),
            }

            # Axe X
            x_field = self.visual_x_field_var.get().split(' - ')[0] if ' - ' in self.visual_x_field_var.get() else self.visual_x_field_var.get()
            data['x_axis'] = {
                'field': x_field,
                'label': self.visual_x_label_var.get(),
            }

            # Axe Y
            data['y_axis'] = {
                'label': self.visual_y_label_var.get(),
            }
            try:
                data['y_axis']['divisor'] = int(self.visual_y_divisor_var.get()) if self.visual_y_divisor_var.get() else 1
            except ValueError:
                data['y_axis']['divisor'] = float(self.visual_y_divisor_var.get()) if self.visual_y_divisor_var.get() else 1

            if self.visual_y_min_var.get():
                try:
                    data['y_axis']['min'] = int(self.visual_y_min_var.get())
                except ValueError:
                    data['y_axis']['min'] = float(self.visual_y_min_var.get())

            # Courbes
            data['curves'] = []
            for w in self.visual_curve_widgets:
                field = w['field'].get().split(' - ')[0] if ' - ' in w['field'].get() else w['field'].get()
                curve = {
                    'field': field,
                    'label': w['label'].get(),
                    'color': w['color'].get(),
                    'style': w['style'].get(),
                }
                if w['fill'].get():
                    curve['fill'] = True
                    try:
                        curve['fill_alpha'] = float(w['fill_alpha'].get())
                    except ValueError:
                        curve['fill_alpha'] = 0.2
                data['curves'].append(curve)

            # Sauvegarder
            yaml_content = yaml.dump(data, allow_unicode=True, default_flow_style=False, sort_keys=False)
            self.current_file_path.write_text(yaml_content, encoding='utf-8')
            self.visual_status_var.set(f"✓ {t('editor.saved')}")

        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_save', error=e))

    def update_graph_files(self, graph_files):
        self.graph_files = graph_files
        values = []
        self._graph_ids = []
        for g_id, g_def in graph_files.items():
            # Support English (title) and French (titre)
            titre = get_yaml_value(g_def, 'title', 'titre', g_id)
            values.append(titre)
            self._graph_ids.append(g_id)
        # Add "New graph..." option at the end
        values.append(t('panel.new_graph'))
        self._graph_ids.append('__NEW_GRAPH__')
        self.graph_combo['values'] = values

    def _on_graph_selected(self, event=None):
        idx = self.graph_combo.current()
        if idx >= 0 and idx < len(self._graph_ids):
            selected_id = self._graph_ids[idx]
            if selected_id == '__NEW_GRAPH__':
                # Reset selection to previous graph
                if self.current_graph_id:
                    prev_idx = self._graph_ids.index(self.current_graph_id) if self.current_graph_id in self._graph_ids else -1
                    if prev_idx >= 0:
                        self.graph_combo.current(prev_idx)
                # Trigger new graph creation
                if self.on_new_graph:
                    self.on_new_graph()
            else:
                self.current_graph_id = selected_id
                self.current_graph_def = self.graph_files.get(self.current_graph_id)
                # Reset zoom/pan quand on change de graphique
                self._zoom_level = 1.0
                self._pan_offset_x = 0
                self._pan_offset_y = 0
                self.redraw()

    def _on_edit_click(self):
        if self.current_graph_id and self.on_edit_graph:
            graph_def = self.graph_files.get(self.current_graph_id)
            if graph_def and '_file' in graph_def:
                self.on_edit_graph(graph_def['_file'])

    def _on_enlarge_click(self):
        if self.current_graph_id and self.on_click_enlarge:
            self.on_click_enlarge(self.panel_index, self.current_graph_id, self.current_graph_def)

    def _on_expand_inplace_click(self):
        if self.current_graph_id and self.on_expand_inplace:
            self.on_expand_inplace(self.panel_index, self.current_graph_id, self.current_graph_def)

    def set_graph(self, graph_id):
        if graph_id in self._graph_ids:
            idx = self._graph_ids.index(graph_id)
            self.graph_combo.current(idx)
            self.current_graph_id = graph_id
            self.current_graph_def = self.graph_files.get(graph_id)
            # Reset zoom/pan quand on change de graphique
            self._zoom_level = 1.0
            self._pan_offset_x = 0
            self._pan_offset_y = 0
            self.redraw()

    def set_data(self, resultats, params, devise="€"):
        self.resultats = resultats
        self.params = params
        self.devise = devise
        self.redraw()

    def update_font_size(self, new_size):
        """Met à jour la taille de police du graphique et de l'éditeur"""
        self.font_size = new_size
        self.editor_text.configure(font=('Consolas', new_size))

    def refresh_labels(self):
        """Met à jour tous les labels avec la langue courante"""
        self.graph_label.configure(text=t('panel.graph_select'))
        # Les boutons de la barre unifiée utilisent des icônes seulement (les tooltips seront mis à jour)

    def _update_curves_panel(self, courbes):
        """Met à jour le panneau de visibilité des courbes"""
        # Skip si c'est un fake panel (utilisé par EnlargedGraphWindow)
        if not hasattr(self, 'content_frame'):
            return

        # Supprimer les anciennes checkboxes
        for widget in self.curve_checkboxes:
            widget.destroy()
        self.curve_checkboxes = []

        # Si moins de 2 courbes, pas besoin de panneau
        if len(courbes) < 2:
            self.curves_frame.pack_forget()
            return

        # Afficher le panneau
        self.curves_frame.pack(fill='x', padx=5, pady=2, before=self.content_frame)

        # Créer une checkbox pour chaque courbe
        for courbe in courbes:
            label = get_yaml_value(courbe, 'label', 'label', '').replace('{devise}', self.devise)
            couleur = get_yaml_value(courbe, 'color', 'couleur', '#000000')

            # Créer la variable si elle n'existe pas
            if label not in self.curve_visibility:
                self.curve_visibility[label] = tk.BooleanVar(value=True)

            # Créer la checkbox
            cb = ttk.Checkbutton(
                self.curves_frame,
                text=label,
                variable=self.curve_visibility[label],
                command=self.redraw
            )
            cb.pack(side='left', padx=5)
            self.curve_checkboxes.append(cb)

    def _on_scroll_up(self, event):
        """Linux scroll up - Ctrl=zoom graphique in, sans Ctrl=zoom image in"""
        if event.state & 0x4:
            self._zoom_graph(0.85)
        else:
            self._zoom_image(1.15)

    def _on_scroll_down(self, event):
        """Linux scroll down - Ctrl=zoom graphique out, sans Ctrl=zoom image out"""
        if event.state & 0x4:
            self._zoom_graph(1.18)
        else:
            self._zoom_image(0.87)

    def _on_mousewheel(self, event):
        """Handle mouse wheel - Ctrl=zoom graphique, sans Ctrl=zoom image"""
        # Vérifier si Ctrl est pressé (state & 0x4 sur Linux/Windows)
        ctrl_pressed = event.state & 0x4

        if ctrl_pressed:
            # Ctrl + molette = zoom sur les données du graphique
            if event.delta > 0:
                self._zoom_graph(0.85)
            else:
                self._zoom_graph(1.18)
        else:
            # Molette seule = zoom sur l'image (loupe)
            if event.delta > 0:
                self._zoom_image(1.15)
            else:
                self._zoom_image(0.87)

    def _zoom_image(self, factor):
        """Zoom sur l'image (comme une loupe)"""
        if not self._graph_image:
            return

        new_zoom = self._zoom_level * factor
        if new_zoom < 0.5 or new_zoom > 4.0:
            return

        self._zoom_level = new_zoom
        self._update_display()

    def _zoom_graph(self, factor):
        """Zoom sur les données du graphique (modifie les limites des axes)"""
        if not hasattr(self, '_original_xlim') or not self._original_xlim:
            return

        xlim = self.ax.get_xlim()
        ylim = self.ax.get_ylim()

        x_center = (xlim[0] + xlim[1]) / 2
        y_center = (ylim[0] + ylim[1]) / 2

        x_half = (xlim[1] - xlim[0]) / 2 * factor
        y_half = (ylim[1] - ylim[0]) / 2 * factor

        orig_x_range = self._original_xlim[1] - self._original_xlim[0]
        orig_y_range = self._original_ylim[1] - self._original_ylim[0]
        if x_half * 2 < orig_x_range * 0.1 or x_half * 2 > orig_x_range * 3:
            return
        if y_half * 2 < orig_y_range * 0.1 or y_half * 2 > orig_y_range * 3:
            return

        self.ax.set_xlim(x_center - x_half, x_center + x_half)
        self.ax.set_ylim(y_center - y_half, y_center + y_half)

        self._render_to_image(reset_view=False)

    def _reset_zoom(self):
        """Remet le zoom image ET les axes à leur état original"""
        # Reset zoom image
        self._zoom_level = 1.0
        self._pan_offset_x = 0
        self._pan_offset_y = 0
        # Reset zoom graphique
        if hasattr(self, '_original_xlim') and self._original_xlim:
            self.ax.set_xlim(self._original_xlim)
            self.ax.set_ylim(self._original_ylim)
            self._render_to_image(reset_view=False)
        else:
            self._update_display()

    def _on_drag_start(self, event):
        """Début du drag pour pan"""
        self._drag_start_x = event.x
        self._drag_start_y = event.y
        # Mémoriser si Ctrl est pressé au début du drag
        self._drag_ctrl = event.state & 0x4
        if self._drag_ctrl:
            self._drag_xlim = self.ax.get_xlim()
            self._drag_ylim = self.ax.get_ylim()

    def _on_drag_motion(self, event):
        """Mouvement pendant le drag - Ctrl=pan graphique, sans Ctrl=pan image"""
        if self._drag_ctrl:
            # Ctrl + drag = pan sur les données du graphique
            if not hasattr(self, '_original_xlim') or not self._original_xlim:
                return
            if not hasattr(self, '_drag_xlim'):
                return

            dx_pixels = event.x - self._drag_start_x
            dy_pixels = event.y - self._drag_start_y

            canvas_width = self.display_canvas.winfo_width()
            canvas_height = self.display_canvas.winfo_height()

            x_range = self._drag_xlim[1] - self._drag_xlim[0]
            y_range = self._drag_ylim[1] - self._drag_ylim[0]

            dx_data = -dx_pixels / canvas_width * x_range
            dy_data = dy_pixels / canvas_height * y_range

            new_xlim = (self._drag_xlim[0] + dx_data, self._drag_xlim[1] + dx_data)
            new_ylim = (self._drag_ylim[0] + dy_data, self._drag_ylim[1] + dy_data)

            self.ax.set_xlim(new_xlim)
            self.ax.set_ylim(new_ylim)

            self._render_to_image(reset_view=False)
        else:
            # Drag seul = pan sur l'image (loupe)
            if not self._graph_image:
                return

            dx = event.x - self._drag_start_x
            dy = event.y - self._drag_start_y

            self._pan_offset_x += dx
            self._pan_offset_y += dy

            self._drag_start_x = event.x
            self._drag_start_y = event.y

            self._update_display()

    def _on_canvas_resize(self, event):
        """Redessine quand le canvas change de taille significativement"""
        if not self.current_graph_def or not self.resultats:
            return

        canvas_width = event.width
        canvas_height = event.height

        # Vérifier si la taille a changé significativement
        if self._graph_image:
            img_w, img_h = self._graph_image.size
            # Si la différence est > 50px, re-rendre
            if abs(canvas_width - img_w) > 50 or abs(canvas_height - img_h) > 50:
                self._render_to_image()
            else:
                self._update_display()
        else:
            # Pas encore d'image, en créer une
            self._render_to_image()

    def _update_display(self):
        """Met à jour l'affichage de l'image avec zoom et pan"""
        if not self._graph_image:
            return

        canvas_width = self.display_canvas.winfo_width()
        canvas_height = self.display_canvas.winfo_height()

        if canvas_width < 10 or canvas_height < 10:
            return

        # Calculer la taille de l'image zoomée
        orig_width, orig_height = self._graph_image.size
        new_width = int(orig_width * self._zoom_level)
        new_height = int(orig_height * self._zoom_level)

        # Redimensionner l'image
        if new_width > 0 and new_height > 0:
            resized = self._graph_image.resize((new_width, new_height), Image.LANCZOS)
            self._photo_image = ImageTk.PhotoImage(resized)

            # Calculer la position (centré + offset pan)
            x = (canvas_width // 2) + self._pan_offset_x
            y = (canvas_height // 2) + self._pan_offset_y

            # Mettre à jour le canvas
            self.display_canvas.delete('all')
            self._canvas_image_id = self.display_canvas.create_image(
                x, y, image=self._photo_image, anchor='center'
            )

    def _generate_save_filename(self):
        """Génère un nom de fichier intelligent: scenario_graphtype_CODE=val_CODE=val"""
        parts = []

        # 1. Nom du scénario
        scenario_name = ""
        if self.get_scenario_name_callback:
            scenario_name = self.get_scenario_name_callback()
        if scenario_name:
            # Nettoyer le nom du scénario (enlever extension et chemin)
            scenario_name = Path(scenario_name).stem.replace(' ', '_')
            parts.append(scenario_name)

        # 2. Type/nom du graphique
        if self.current_graph_id:
            parts.append(self.current_graph_id)

        # 3. Paramètres modifiés (code=valeur)
        if self.get_modified_params_callback:
            modified = self.get_modified_params_callback()
            for param_name, value in modified.items():
                code = PARAM_NAMES_TO_CODES.get(param_name)
                if code:
                    # Formater la valeur (entier si possible, sinon 1 décimale)
                    if isinstance(value, float) and value == int(value):
                        val_str = str(int(value))
                    elif isinstance(value, float):
                        val_str = f"{value:.1f}".rstrip('0').rstrip('.')
                    else:
                        val_str = str(value)
                    parts.append(f"{code}={val_str}")

        # Assembler le nom de fichier
        if not parts:
            return "graphique"
        return "_".join(parts)

    def _save_graph(self):
        """Sauvegarde le graphique actuel en SVG ou PNG"""
        if not self.current_graph_id or not self.resultats:
            messagebox.showwarning(t('dialogs.warning'), t('dialogs.no_graph_to_save'))
            return

        # Générer le nom de fichier par défaut
        default_name = self._generate_save_filename()

        # Dialogue de sauvegarde
        file_path = filedialog.asksaveasfilename(
            title=t('dialogs.save_graph_title'),
            initialfile=default_name,
            defaultextension=".svg",
            filetypes=[
                (t('file_types.svg'), "*.svg"),
                (t('file_types.png'), "*.png"),
                (t('file_types.pdf'), "*.pdf"),
                (t('file_types.jpeg'), "*.jpg"),
                (t('file_types.all'), "*.*")
            ]
        )

        if not file_path:
            return

        try:
            # Sauvegarder avec la bonne extension
            self.fig.savefig(file_path, dpi=150, bbox_inches='tight',
                           facecolor='white', edgecolor='none')
            self._update_status(t('status.graph_saved', path=file_path))
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_save', error=e))

    def _copy_graph_png(self):
        """Copie le graphique dans le presse-papier en PNG"""
        if not self.current_graph_id or not self.resultats:
            self._update_status(t('dialogs.no_graph_to_copy'), error=True)
            return

        try:
            # Sauvegarder en PNG dans un buffer
            buf = io.BytesIO()
            self.fig.savefig(buf, format='png', dpi=150, bbox_inches='tight',
                           facecolor='white', edgecolor='none')
            buf.seek(0)
            png_data = buf.getvalue()

            # Copier dans le presse-papier selon le système
            system = platform.system()
            if system == 'Linux':
                # Vérifier wl-copy (Wayland) ou xclip (X11)
                wl_copy_path = shutil.which('wl-copy')
                xclip_path = shutil.which('xclip')

                if wl_copy_path:
                    # Wayland: wl-copy est non-bloquant
                    process = subprocess.Popen(
                        ['wl-copy', '--type', 'image/png'],
                        stdin=subprocess.PIPE,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL
                    )
                    process.communicate(png_data, timeout=5)
                elif xclip_path:
                    # X11: xclip - lancer en arrière-plan et ne pas attendre
                    # xclip garde un daemon pour servir le clipboard
                    import tempfile
                    with tempfile.NamedTemporaryFile(suffix='.png', delete=False) as tmp:
                        tmp.write(png_data)
                        tmp_path = tmp.name

                    # Redirection stdin < fichier (méthode canonique pour images binaires)
                    subprocess.Popen(
                        f'xclip -selection clipboard -t image/png < "{tmp_path}" && sleep 60 && rm -f "{tmp_path}"',
                        shell=True,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL,
                        start_new_session=True
                    )
                else:
                    self._update_status("wl-copy ou xclip requis", error=True)
                    return

            elif system == 'Darwin':  # macOS
                # Utiliser osascript pour macOS
                import tempfile
                with tempfile.NamedTemporaryFile(suffix='.png', delete=False) as tmp:
                    tmp.write(png_data)
                    tmp_path = tmp.name
                subprocess.run(['osascript', '-e',
                              f'set the clipboard to (read (POSIX file "{tmp_path}") as «class PNGf»)'],
                              check=True, timeout=5)
                os.unlink(tmp_path)
            elif system == 'Windows':
                # Windows: utiliser win32clipboard
                try:
                    import win32clipboard
                    from PIL import Image
                    img = Image.open(io.BytesIO(png_data))
                    output = io.BytesIO()
                    img.convert('RGB').save(output, 'BMP')
                    data = output.getvalue()[14:]  # Skip BMP header
                    output.close()

                    win32clipboard.OpenClipboard()
                    win32clipboard.EmptyClipboard()
                    win32clipboard.SetClipboardData(win32clipboard.CF_DIB, data)
                    win32clipboard.CloseClipboard()
                except ImportError:
                    self._update_status("pywin32 requis (pip install pywin32)", error=True)
                    return

            self._update_status(t('status.graph_copied_png'))
        except subprocess.TimeoutExpired:
            self._update_status("Timeout copie PNG", error=True)
        except Exception as e:
            self._update_status(f"Erreur copie: {e}", error=True)

    def _copy_graph_svg(self):
        """Copie le graphique dans le presse-papier en SVG (texte)"""
        if not self.current_graph_id or not self.resultats:
            self._update_status(t('dialogs.no_graph_to_copy'), error=True)
            return

        try:
            # Sauvegarder en SVG dans un buffer
            buf = io.BytesIO()
            self.fig.savefig(buf, format='svg', bbox_inches='tight',
                           facecolor='white', edgecolor='none')
            buf.seek(0)
            svg_content = buf.getvalue().decode('utf-8')

            # Copier dans le presse-papier selon le système
            system = platform.system()
            if system == 'Linux':
                # Vérifier wl-copy (Wayland) ou xclip (X11)
                wl_copy_path = shutil.which('wl-copy')
                xclip_path = shutil.which('xclip')

                if wl_copy_path:
                    # Wayland: wl-copy
                    process = subprocess.Popen(
                        ['wl-copy'],
                        stdin=subprocess.PIPE,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL
                    )
                    process.communicate(svg_content.encode('utf-8'), timeout=5)
                elif xclip_path:
                    # X11: xclip en background
                    import tempfile
                    with tempfile.NamedTemporaryFile(mode='w', suffix='.svg', delete=False, encoding='utf-8') as tmp:
                        tmp.write(svg_content)
                        tmp_path = tmp.name

                    subprocess.Popen(
                        f'xclip -selection clipboard -i "{tmp_path}" && sleep 60 && rm -f "{tmp_path}"',
                        shell=True,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL,
                        start_new_session=True
                    )
                else:
                    # Fallback tkinter
                    self.master.clipboard_clear()
                    self.master.clipboard_append(svg_content)
                    self.master.update()
            elif system == 'Darwin':  # macOS
                # pbcopy pour macOS
                process = subprocess.Popen(['pbcopy'], stdin=subprocess.PIPE)
                process.communicate(svg_content.encode('utf-8'), timeout=5)
            elif system == 'Windows':
                # Utiliser tkinter pour Windows
                self.master.clipboard_clear()
                self.master.clipboard_append(svg_content)
                self.master.update()
            else:
                # Fallback pour autres systèmes
                self.master.clipboard_clear()
                self.master.clipboard_append(svg_content)
                self.master.update()

            self._update_status(t('status.graph_copied_svg'))
        except subprocess.TimeoutExpired:
            self._update_status("Timeout copie SVG", error=True)
        except Exception as e:
            self._update_status(f"Erreur copie: {e}", error=True)

    def redraw(self):
        self.ax.clear()

        if not self.current_graph_def:
            self.ax.text(0.5, 0.5, t('graph_panel.select_graph'),
                        ha='center', va='center', transform=self.ax.transAxes)
            self._render_to_image()
            return

        if not self.resultats:
            self.ax.text(0.5, 0.5, t('graph_panel.loading'),
                        ha='center', va='center', transform=self.ax.transAxes)
            self._render_to_image()
            return

        graph_def = self.current_graph_def
        graph_type = graph_def.get('type', 'line')

        try:
            if graph_type == 'line' or graph_type == 'area':
                self._draw_line_graph(graph_def)
            elif graph_type == 'bar':
                self._draw_bar_graph(graph_def)
            elif graph_type == 'stacked_bar':
                self._draw_stacked_bar_graph(graph_def)
            elif graph_type == 'salary_evolution':
                self._draw_salary_evolution_graph(graph_def)
            else:
                self._draw_line_graph(graph_def)
        except Exception as e:
            self.ax.text(0.5, 0.5, f"Erreur: {e}",
                        ha='center', va='center', transform=self.ax.transAxes,
                        color='red')

        # Support English (title) and French (titre)
        titre = get_yaml_value(graph_def, 'title', 'titre', '').replace('{devise}', self.devise)
        self.ax.set_title(titre, fontsize=self.font_size)
        self.ax.grid(True, alpha=0.3)

        # Ajuster l'axe X pour réduire la marge gauche automatique de matplotlib
        xlim = self.ax.get_xlim()
        if graph_type in ('bar', 'stacked_bar'):
            # Pour les histogrammes: garder une petite marge pour la première barre
            # (barres centrées sur x=0,1,2... avec largeur 0.6)
            self.ax.set_xlim(left=-0.4)
        elif xlim[0] < 5 and xlim[0] > -5:
            # Pour les graphiques en ligne commençant proche de 0
            self.ax.set_xlim(left=0)

        # Appliquer les tailles de police aux axes
        self.ax.tick_params(axis='both', labelsize=self.font_size - 2)

        self.fig.tight_layout()
        self._render_to_image()

    def _render_to_image(self, reset_view=True):
        """Rend le graphique matplotlib dans une image PIL et l'affiche"""
        # Obtenir la taille du canvas pour adapter la taille de la figure
        canvas_width = self.display_canvas.winfo_width()
        canvas_height = self.display_canvas.winfo_height()

        if canvas_width < 50 or canvas_height < 50:
            # Canvas pas encore initialisé, utiliser des valeurs par défaut
            canvas_width = 400
            canvas_height = 300

        # Ajuster la taille de la figure pour remplir le canvas
        dpi = 100
        fig_width = canvas_width / dpi
        fig_height = canvas_height / dpi
        self.fig.set_size_inches(fig_width, fig_height)

        # Ajuster les marges pour garder l'alignement des axes
        self.fig.tight_layout(pad=0.5)

        # Rendre dans un buffer PNG (sans bbox_inches='tight' pour préserver l'alignement)
        buf = io.BytesIO()
        self.fig.savefig(buf, format='png', dpi=dpi, facecolor='white',
                         edgecolor='none')
        buf.seek(0)

        # Charger comme image PIL
        self._graph_image = Image.open(buf).copy()
        buf.close()

        # Reset zoom/pan seulement si demandé (pas lors des zoom/pan interactifs)
        if reset_view:
            self._zoom_level = 1.0
            self._pan_offset_x = 0
            self._pan_offset_y = 0
            # Sauvegarder les limites d'axes originales pour le zoom/pan
            self._original_xlim = self.ax.get_xlim()
            self._original_ylim = self.ax.get_ylim()

        # Afficher l'image
        self._update_display()

    def get_ylim(self):
        """Retourne les limites actuelles de l'axe Y (ymin, ymax)"""
        if self.ax:
            return self.ax.get_ylim()
        return (0, 1)

    def set_ylim(self, ymin, ymax):
        """Définit les limites de l'axe Y et met à jour l'affichage"""
        if self.ax and ymin is not None and ymax is not None:
            self.ax.set_ylim(ymin, ymax)
            self._original_ylim = (ymin, ymax)
            self._render_to_image(reset_view=False)

    def _draw_line_graph(self, graph_def):
        # Support English (x_axis/y_axis/curves) and French (axe_x/axe_y/courbes)
        axe_x = get_yaml_value(graph_def, 'x_axis', 'axe_x', {})
        axe_y = get_yaml_value(graph_def, 'y_axis', 'axe_y', {})
        courbes = get_yaml_value(graph_def, 'curves', 'courbes', [])

        # Mettre à jour le panneau de visibilité des courbes
        self._update_curves_panel(courbes)

        # Résoudre le code 4 lettres ou utiliser le nom complet
        x_champ = resolve_field_code(get_yaml_value(axe_x, 'field', 'champ', 'annee'))
        x_data = [r.get(x_champ, 0) for r in self.resultats]
        diviseur_y = get_yaml_value(axe_y, 'divisor', 'diviseur', 1)

        # Collecter les valeurs Y des courbes visibles pour recalculer l'axe Y
        all_y_values = []
        visible_count = 0

        for courbe in courbes:
            y_champ_raw = get_yaml_value(courbe, 'field', 'champ')
            if not y_champ_raw:
                continue
            y_champ = resolve_field_code(y_champ_raw)

            y_data = [r.get(y_champ, 0) / diviseur_y for r in self.resultats]
            couleur = get_yaml_value(courbe, 'color', 'couleur', COLORS['primary'])
            label = get_yaml_value(courbe, 'label', 'label', y_champ).replace('{devise}', self.devise)
            style = get_yaml_value(courbe, 'style', 'style', 'solid')
            linestyle = {'solid': '-', 'dashed': '--', 'dotted': ':'}
            ls = linestyle.get(style, '-')

            # Vérifier si la courbe est visible
            is_visible = self.curve_visibility.get(label, tk.BooleanVar(value=True)).get()
            if not is_visible:
                continue

            visible_count += 1
            all_y_values.extend(y_data)

            self.ax.plot(x_data, y_data, label=label, color=couleur, linestyle=ls, linewidth=2)

            if get_yaml_value(courbe, 'fill', 'remplir', False):
                alpha = get_yaml_value(courbe, 'fill_alpha', 'alpha_remplissage', 0.2)
                self.ax.fill_between(x_data, y_data, alpha=alpha, color=couleur)

        xlabel = get_yaml_value(axe_x, 'label', 'label', '').replace('{devise}', self.devise)
        ylabel = get_yaml_value(axe_y, 'label', 'label', '').replace('{devise}', self.devise)
        self.ax.set_xlabel(xlabel, fontsize=self.font_size - 2)
        self.ax.set_ylabel(ylabel, fontsize=self.font_size - 2)

        # Recalculer l'axe Y selon les courbes visibles
        if all_y_values:
            y_min = min(all_y_values)
            y_max = max(all_y_values)
            # Ajouter une marge de 5%
            margin = (y_max - y_min) * 0.05 if y_max != y_min else 1
            if axe_y.get('min') is not None:
                self.ax.set_ylim(bottom=axe_y['min'], top=y_max + margin)
            else:
                self.ax.set_ylim(bottom=y_min - margin, top=y_max + margin)

        if visible_count > 1:
            self.ax.legend(fontsize=self.font_size - 4)

    def _draw_bar_graph(self, graph_def):
        if not self.params:
            return
        self._update_curves_panel([])  # Pas de checkboxes pour barres simples
        self._compute_salary_data(graph_def)

    def _draw_stacked_bar_graph(self, graph_def):
        if not self.params:
            return
        self._compute_salary_data(graph_def, stacked=True)

    def _draw_salary_evolution_graph(self, graph_def):
        """Dessine l'évolution du pouvoir d'achat par niveau de salaire au fil des ans

        Supporte:
        - Phases: zones colorées délimitant des périodes
        - Milestones: lignes verticales marquant des événements
        - Annotations: textes avec boîtes optionnelles
        - Valeurs finales: affichage des valeurs en fin de courbe
        """
        if not self.params or not self.resultats:
            return
        self.curves_frame.pack_forget()  # Pas de checkboxes pour salary_evolution

        axe_y = graph_def.get('y_axis', graph_def.get('axe_y', {}))
        mode = axe_y.get('mode', 'absolute')  # absolute, gain_absolute, gain_percent
        salaires_liste = graph_def.get('salaries', graph_def.get('salaires', self.params.salaires_analyses))
        show_baseline = graph_def.get('show_baseline', False)
        show_final_values = graph_def.get('show_final_values', False)
        final_value_decimals = graph_def.get('final_value_decimals', 1)

        assurances_total = (self.params.assurance_sante + self.params.assurance_chomage +
                          self.params.assurance_pension + self.params.assurance_education)

        # Couleurs configurables (défaut si non spécifié)
        default_colors = ['#FF7F7F', '#FFB347', '#90EE90', '#87CEEB', '#DDA0DD', '#2F4F4F']
        colors = graph_def.get('colors', default_colors)
        base_linewidth = graph_def.get('linewidth', 2)
        linewidth_max = graph_def.get('linewidth_max_salary', base_linewidth)

        # === PHASES: zones colorées ===
        phases = graph_def.get('phases', [])
        for phase in phases:
            start_year = self._resolve_phase_boundary(phase.get('start', 0))
            end_year = self._resolve_phase_boundary(phase.get('end', len(self.resultats)))
            if start_year is not None and end_year is not None:
                phase_color = phase.get('color', '#B8D4E8')
                phase_alpha = phase.get('alpha', 0.3)
                phase_label = phase.get('label', '')
                self.ax.axvspan(start_year, end_year, alpha=phase_alpha, color=phase_color, label=phase_label)

        # === MILESTONES: lignes verticales ===
        milestones = graph_def.get('milestones', [])
        for milestone in milestones:
            year = milestone.get('year')
            if year is None and 'auto' in milestone:
                year = self._resolve_auto_condition(milestone['auto'])
            if year is not None:
                ms_color = milestone.get('color', '#6699CC')
                ms_style = {'solid': '-', 'dashed': '--', 'dotted': ':'}.get(milestone.get('style', 'dashed'), '--')
                ms_linewidth = milestone.get('linewidth', 1.5)
                self.ax.axvline(x=year, color=ms_color, linestyle=ms_style, linewidth=ms_linewidth, alpha=0.8)

                # Label du milestone
                ms_label = milestone.get('label', '')
                if ms_label:
                    label_pos = milestone.get('label_position', 'bottom')
                    y_pos = 0.05 if label_pos == 'bottom' else (0.95 if label_pos == 'top' else 0.5)
                    self.ax.text(year, self.ax.get_ylim()[0] + (self.ax.get_ylim()[1] - self.ax.get_ylim()[0]) * y_pos,
                               ms_label, ha='center', va='bottom', fontsize=self.font_size - 4,
                               color=ms_color, style='italic')

        # === COURBES DE SALAIRES ===
        all_curves_data = []  # Pour stocker les données des courbes pour les valeurs finales

        # Récupérer le différentiel de l'année 1 (premier année de transition)
        # pour l'utiliser à l'année 0 si celle-ci n'a pas de différentiel
        diff_annee1 = 0
        for r in self.resultats:
            if r.get('annee', 0) == 1:
                diff_annee1 = r.get('differentiel_pct', 0) / 100
                break

        for idx, salaire in enumerate(salaires_liste):
            net_actuel = salaire * (1 - self.params.prelevement_actuel_total)
            taux_ti = self.params.taxes_indirectes.get(salaire, 0.10)
            gain_taxes = net_actuel * taux_ti

            annees = []
            valeurs = []

            for resultat in self.resultats:
                annee = resultat.get('annee', 0)
                diff_pct = resultat.get('differentiel_pct', 0) / 100

                # À l'année 0, utiliser le différentiel de l'année 1 pour cohérence
                # avec le graphique "Effet combiné année 0"
                if annee == 0 and diff_pct == 0 and diff_annee1 > 0:
                    diff_pct = diff_annee1

                base_imposable = max(0, salaire - self.params.abattement_forfaitaire)
                flat_tax = base_imposable * self.params.taux_flat_tax
                differentiel = salaire * diff_pct
                net_nouveau = salaire - flat_tax - differentiel - assurances_total + gain_taxes

                if mode == 'absolute':
                    valeur = net_nouveau
                elif mode == 'gain_absolute':
                    valeur = net_nouveau - net_actuel
                elif mode == 'gain_percent':
                    valeur = ((net_nouveau - net_actuel) / net_actuel * 100) if net_actuel > 0 else 0
                else:
                    valeur = net_nouveau

                annees.append(annee)
                valeurs.append(valeur)

            color = colors[idx % len(colors)]
            # Ligne plus épaisse pour le salaire le plus élevé
            lw = linewidth_max if idx == len(salaires_liste) - 1 else base_linewidth
            self.ax.plot(annees, valeurs, label=f"{salaire}{self.devise}", color=color, linewidth=lw)
            all_curves_data.append((annees, valeurs, color, salaire))

        # Aligner l'axe X pour que l'année 0 soit sur l'axe vertical
        if annees:
            self.ax.set_xlim(left=0)

        # Ligne de référence (salaire actuel)
        if show_baseline and mode == 'absolute':
            baseline_label = graph_def.get('baseline_label', 'Net actuel')
            for idx, salaire in enumerate(salaires_liste):
                net_actuel = salaire * (1 - self.params.prelevement_actuel_total)
                color = colors[idx % len(colors)]
                self.ax.axhline(y=net_actuel, color=color, linestyle='--', alpha=0.3)

        # === VALEURS FINALES ===
        if show_final_values and all_curves_data:
            for annees, valeurs, color, salaire in all_curves_data:
                if valeurs:
                    final_val = valeurs[-1]
                    final_year = annees[-1]
                    suffix = '%' if mode == 'gain_percent' else self.devise
                    self.ax.annotate(f'{final_val:.{final_value_decimals}f}{suffix}',
                                   xy=(final_year, final_val),
                                   xytext=(5, 0), textcoords='offset points',
                                   fontsize=self.font_size - 4, color=color,
                                   fontweight='bold', va='center')

        # === ANNOTATIONS ===
        annotations = graph_def.get('annotations', [])
        for annot in annotations:
            text = annot.get('text', '')
            if not text:
                continue

            # Position absolue ou relative
            if 'x_rel' in annot and 'y_rel' in annot:
                transform = self.ax.transAxes
                x_pos = annot['x_rel']
                y_pos = annot['y_rel']
            else:
                transform = self.ax.transData
                x_pos = annot.get('x', 0)
                y_pos = annot.get('y', 0)

            annot_kwargs = {
                'fontsize': annot.get('fontsize', self.font_size - 2),
                'fontweight': annot.get('fontweight', 'normal'),
                'ha': annot.get('ha', 'left'),
                'va': annot.get('va', 'top'),
                'transform': transform
            }

            if annot.get('bbox', False):
                annot_kwargs['bbox'] = dict(
                    boxstyle='round,pad=0.3',
                    facecolor=annot.get('bbox_color', '#FFFFCC'),
                    alpha=annot.get('bbox_alpha', 0.8),
                    edgecolor='none'
                )

            self.ax.text(x_pos, y_pos, text, **annot_kwargs)

        # Labels
        xlabel = get_yaml_value(graph_def.get('x_axis', graph_def.get('axe_x', {})), 'label', 'label', 'Année')
        ylabel = get_yaml_value(axe_y, 'label', 'label', 'Salaire net').replace('{devise}', self.devise)
        self.ax.set_xlabel(xlabel, fontsize=self.font_size - 2)
        self.ax.set_ylabel(ylabel, fontsize=self.font_size - 2)

        # Y-axis min/max from config
        if axe_y.get('min') is not None:
            self.ax.set_ylim(bottom=axe_y['min'])
        if axe_y.get('max') is not None:
            self.ax.set_ylim(top=axe_y['max'])

        self.ax.legend(fontsize=self.font_size - 4, loc='lower right')

    def _resolve_phase_boundary(self, boundary):
        """Résout une limite de phase (année fixe ou condition auto)"""
        if isinstance(boundary, int) or isinstance(boundary, float):
            return boundary
        if isinstance(boundary, str) and boundary.startswith('auto:'):
            condition = boundary[5:]  # Enlever 'auto:'
            return self._resolve_auto_condition(condition)
        return None

    def _resolve_auto_condition(self, condition):
        """Résout une condition auto comme 'dette_publique<=0' pour trouver l'année"""
        # Parser la condition (ex: "dette_publique<=0", "differentiel_pct==0")
        import re
        match = re.match(r'(\w+)(==|<=|>=|<|>)(-?\d+\.?\d*)', condition)
        if not match:
            return None

        field_name, operator, value = match.groups()
        value = float(value)

        # Résoudre le code 4 lettres si nécessaire
        field_name = resolve_field_code(field_name)

        for resultat in self.resultats:
            field_value = resultat.get(field_name, None)
            if field_value is None:
                continue

            if operator == '==' and field_value == value:
                return resultat.get('annee', 0)
            elif operator == '<=' and field_value <= value:
                return resultat.get('annee', 0)
            elif operator == '>=' and field_value >= value:
                return resultat.get('annee', 0)
            elif operator == '<' and field_value < value:
                return resultat.get('annee', 0)
            elif operator == '>' and field_value > value:
                return resultat.get('annee', 0)

        return None

    def _compute_salary_data(self, graph_def, stacked=False):
        salaires = self.params.salaires_analyses
        assurances_total = (self.params.assurance_sante + self.params.assurance_chomage +
                          self.params.assurance_pension + self.params.assurance_education)
        graph_id = self.current_graph_id or ''

        if 'an0' in graph_id or 'annee0' in graph_id.lower():
            annee_0 = next((r for r in self.resultats if r['annee'] == 1), None)
            if not annee_0:
                return
            diff_pct = annee_0['differentiel_pct'] / 100
        else:
            diff_pct = 0

        impacts, gains_ti, effets_nets, pourcentages = [], [], [], []

        for salaire in salaires:
            net_actuel = salaire * (1 - self.params.prelevement_actuel_total)
            base_imposable = max(0, salaire - self.params.abattement_forfaitaire)
            flat_tax = base_imposable * self.params.taux_flat_tax
            differentiel = salaire * diff_pct
            net_nouveau = salaire - flat_tax - differentiel - assurances_total

            impact = round(net_nouveau - net_actuel)
            impacts.append(impact)

            taux_ti = self.params.taxes_indirectes.get(salaire, 0.10)
            gain_ti = round(net_actuel * taux_ti)
            gains_ti.append(gain_ti)
            total = impact + gain_ti
            effets_nets.append(total)
            # Calculer le pourcentage du salaire NET initial (pas brut!)
            pct = (total / net_actuel) * 100 if net_actuel > 0 else 0
            pourcentages.append(pct)

        x = np.arange(len(salaires))
        width = 0.6
        labels = [f"{s}{self.devise}" for s in salaires]

        # Tailles de police pour les annotations
        annot_size = max(10, self.font_size - 6)
        annot_size_big = max(11, self.font_size - 4)
        annot_size_small = max(8, self.font_size - 8)

        # Labels pour les composantes
        label_nouveau = f'Nouveau système* (ass. {assurances_total:.0f}{self.devise} incl.)'
        label_taxes = 'Taxes indirectes abolies'

        if stacked:
            # Créer les checkboxes pour les deux composantes
            courbes_factices = [
                {'label': label_nouveau},
                {'label': label_taxes}
            ]
            self._update_curves_panel(courbes_factices)

            # Vérifier la visibilité de chaque composante
            show_nouveau = self.curve_visibility.get(label_nouveau, tk.BooleanVar(value=True)).get()
            show_taxes = self.curve_visibility.get(label_taxes, tk.BooleanVar(value=True)).get()

            # Collecter les valeurs pour recalculer l'axe Y
            all_y_values = [0]  # Au moins 0 pour la ligne de base
            current_bottom = [0] * len(salaires)

            if show_nouveau:
                self.ax.bar(x, impacts, width, color=COLORS['warning'], alpha=0.85, label=label_nouveau)
                all_y_values.extend(impacts)
                current_bottom = impacts[:]
                # Annotations pour nouveau système
                for i, imp in enumerate(impacts):
                    self.ax.annotate(f'+{imp}{self.devise}', (i, imp/2), ha='center', va='center',
                                   fontsize=annot_size, fontweight='bold')

            if show_taxes:
                bottom = current_bottom if show_nouveau else None
                self.ax.bar(x, gains_ti, width, bottom=bottom, color=COLORS['success'], alpha=0.85, label=label_taxes)
                if show_nouveau:
                    all_y_values.extend([i + g for i, g in zip(impacts, gains_ti)])
                    # Annotations pour taxes
                    for i, (imp, gti) in enumerate(zip(impacts, gains_ti)):
                        self.ax.annotate(f'+{gti}{self.devise}', (i, imp + gti/2), ha='center', va='center',
                                       fontsize=annot_size, fontweight='bold')
                else:
                    all_y_values.extend(gains_ti)
                    for i, gti in enumerate(gains_ti):
                        self.ax.annotate(f'+{gti}{self.devise}', (i, gti/2), ha='center', va='center',
                                       fontsize=annot_size, fontweight='bold')

            # Annotations totales seulement si les deux sont visibles
            if show_nouveau and show_taxes:
                for i, (total, pct) in enumerate(zip(effets_nets, pourcentages)):
                    self.ax.annotate(f'= +{total}{self.devise}\n(+{pct:.1f}%)', (i, total + 30), ha='center',
                                   fontsize=annot_size_big, fontweight='bold', color=COLORS['dark'])

            # Recalculer l'axe Y (respecter min: 0 si spécifié)
            if all_y_values:
                y_max = max(all_y_values)
                margin = y_max * 0.15 if y_max > 0 else 100
                axe_y = graph_def.get('y_axis', graph_def.get('axe_y', {}))
                y_min = axe_y.get('min', None)
                if y_min is not None:
                    self.ax.set_ylim(bottom=y_min, top=y_max + margin)
                else:
                    self.ax.set_ylim(bottom=min(0, min(all_y_values)) - 50, top=y_max + margin)

        else:
            self._update_curves_panel([])  # Pas de checkboxes pour barres simples
            self.ax.bar(x, effets_nets, width, color=COLORS['success'], alpha=0.85)
            for i, (total, pct) in enumerate(zip(effets_nets, pourcentages)):
                # Afficher valeur + pourcentage du salaire
                self.ax.annotate(f'+{total}{self.devise}\n(+{pct:.1f}%)', (i, total + 20), ha='center',
                               fontsize=annot_size_big, fontweight='bold')

        self.ax.axhline(y=0, color=COLORS['dark'], linewidth=1)
        self.ax.set_xticks(x)
        self.ax.set_xticklabels(labels, fontsize=self.font_size - 4)
        self.ax.set_xlabel('Salaire brut', fontsize=self.font_size - 2)
        self.ax.set_ylabel(f'Gain mensuel ({self.devise})', fontsize=self.font_size - 2)

        if stacked and (show_nouveau or show_taxes):
            self.ax.legend(fontsize=self.font_size - 6, loc='upper left')


class ParameterPanel(ttk.Frame):
    """Panneau des paramètres FIXE (pas de scroll) avec mode lecture seule, couleurs et reset"""

    # Paramètres en pourcentage avec leur incrément
    PERCENT_PARAMS = {
        'taux_croissance_base': 0.1,      # +/- 0.1%
        'reduction_solidaire': 1.0,        # +/- 1%
        'differentiel_initial': 0.5,       # +/- 0.5%
        'taux_flat_tax': 0.5,              # +/- 0.5%
    }

    def __init__(self, parent, on_param_change, font_size=11, on_edit_scenario=None, on_scenario_reload=None):
        super().__init__(parent)
        self.on_param_change = on_param_change
        self.on_edit_scenario = on_edit_scenario
        self.on_scenario_reload = on_scenario_reload
        self.param_vars = {}
        self.param_entries = {}
        self.param_reset_btns = {}
        self.param_plus_btns = {}
        self.param_minus_btns = {}
        self.param_labels = {}
        self.original_values = {}
        self.editable = False
        self.font_size = font_size
        self.section_frames = []
        self.scenario_file_path = None
        self.scenario_editor_mode = False
        self._debounce_timer = None  # Timer pour debounce des boutons +/-

        self._create_widgets()

    def _create_widgets(self):
        # Header avec checkbox cadenas et bouton crayon pour éditer scénario
        self.header_frame = ttk.Frame(self)
        self.header_frame.pack(fill='x', padx=5, pady=3)

        self.edit_var = tk.BooleanVar(value=False)
        self.edit_check = ttk.Checkbutton(
            self.header_frame,
            text="🔒",
            variable=self.edit_var,
            command=self._toggle_edit_mode
        )
        self.edit_check.pack(side='left')

        # Petit bouton crayon pour éditer le fichier scénario
        self.edit_scenario_btn = ttk.Button(
            self.header_frame,
            text="✎",
            width=3,
            command=self._on_edit_scenario_click
        )
        self.edit_scenario_btn.pack(side='right', padx=2)

        # Champ devise éditable
        devise_frame = ttk.Frame(self.header_frame)
        devise_frame.pack(side='right', padx=10)
        ttk.Label(devise_frame, text="Devise:").pack(side='left')
        self.devise_var = tk.StringVar(value="€")
        self.devise_entry = ttk.Entry(devise_frame, textvariable=self.devise_var, width=4)
        self.devise_entry.pack(side='left', padx=2)
        self.devise_entry.bind('<KeyRelease>', self._on_devise_change)
        self.devise_entry.configure(state='disabled')

        # Frame FIXE pour les paramètres (pas de scroll!)
        self.params_container = tk.Frame(self, bg=COLORS['readonly_bg'])
        self.params_container.pack(fill='x', padx=5, pady=3)

        # Store section keys for refresh
        self.section_keys = {}

        self._add_section("economy", [
            ('pib_initial', 850.0),
            ('dette_publique_initiale', 884.0),
            ('taux_croissance_base', 1.5),
        ])

        self._add_section("pensions", [
            ('dette_implicite_initiale', 2100.0),
            ('reduction_solidaire', 10.0),
            ('pension_moyenne_annuelle', 15000.0),
        ])

        self._add_section("financing", [
            ('privatisations_totales', 195.0),
            ('differentiel_initial', 10.0),
            ('duree_decroissance_differentiel', 40),
            ('surplus_budgetaire_minimal_pct_pib', 2.0),
            ('surplus_max_pour_dette_transition_pct', 100.0),
        ])

        self._add_section("new_system", [
            ('taux_flat_tax', 25.0),
            ('abattement_forfaitaire', 500.0),
            ('assurance_sante', 73.0),
            ('assurance_chomage', 37.0),
            ('assurance_pension', 59.0),
            ('assurance_education', 46.0),
        ])

        self._update_colors()

        # === EDITEUR DE SCÉNARIO (caché par défaut) ===
        self.scenario_editor_frame = ttk.Frame(self)
        # Ne pas pack maintenant - sera affiché quand on bascule en mode édition

        # Barre d'outils éditeur scénario
        scenario_toolbar = ttk.Frame(self.scenario_editor_frame)
        scenario_toolbar.pack(fill='x', padx=2, pady=2)

        # Boutons icônes uniquement
        self.save_scenario_btn = ttk.Button(scenario_toolbar, text="💾", width=3, command=self._save_scenario_editor)
        self.save_scenario_btn.pack(side='left', padx=2)
        self.validate_scenario_btn = ttk.Button(scenario_toolbar, text="✓", width=3, command=self._validate_scenario_editor)
        self.validate_scenario_btn.pack(side='left', padx=2)

        # Nom du fichier scénario
        self.scenario_name_label = ttk.Label(scenario_toolbar, text="")
        self.scenario_name_label.pack(side='left', padx=10)

        # Checkbox cadenas pour verrouillage (synchronisée avec edit_var)
        self.scenario_lock_check = ttk.Checkbutton(
            scenario_toolbar,
            text="🔓",
            variable=self.edit_var,
            command=self._on_scenario_lock_toggle
        )
        self.scenario_lock_check.pack(side='left', padx=5)

        self.close_scenario_btn = ttk.Button(scenario_toolbar, text="✕", width=3, command=self._close_scenario_editor)
        self.close_scenario_btn.pack(side='right', padx=2)

        self.scenario_editor_status = tk.StringVar(value="")
        ttk.Label(scenario_toolbar, textvariable=self.scenario_editor_status).pack(side='right', padx=5)

        # Zone de texte éditeur scénario (prend toute la place disponible)
        scenario_text_frame = ttk.Frame(self.scenario_editor_frame)
        scenario_text_frame.pack(fill='both', expand=True, padx=2, pady=2)

        self.scenario_editor_text = tk.Text(scenario_text_frame, wrap='none', font=('Consolas', self.font_size))
        scenario_scroll_y = ttk.Scrollbar(scenario_text_frame, orient='vertical', command=self.scenario_editor_text.yview)
        scenario_scroll_x = ttk.Scrollbar(scenario_text_frame, orient='horizontal', command=self.scenario_editor_text.xview)
        self.scenario_editor_text.configure(yscrollcommand=scenario_scroll_y.set, xscrollcommand=scenario_scroll_x.set)

        scenario_scroll_y.pack(side='right', fill='y')
        scenario_scroll_x.pack(side='bottom', fill='x')
        self.scenario_editor_text.pack(fill='both', expand=True, padx=2, pady=2)

        # Binding Select All (Ctrl+A)
        self.scenario_editor_text.bind('<Control-a>', self._select_all_scenario_text)
        self.scenario_editor_text.bind('<Control-A>', self._select_all_scenario_text)

        # Initialiser l'état du cadenas et de l'éditeur
        self._update_scenario_editor_state()

    def _select_all_scenario_text(self, event):
        """Sélectionne tout le texte dans l'éditeur de scénario"""
        self.scenario_editor_text.tag_add('sel', '1.0', 'end')
        self.scenario_editor_text.mark_set('insert', 'end')
        return 'break'

    def _add_section(self, section_key, params):
        # Frame de section SANS scroll - use translation for title
        title = t(f'param_sections.{section_key}')
        frame = tk.LabelFrame(self.params_container, text=title,
                             bg=COLORS['readonly_bg'],
                             font=('TkDefaultFont', self.font_size, 'bold'))
        frame.pack(fill='x', padx=3, pady=2)
        self.section_frames.append(frame)
        self.section_keys[frame] = section_key

        for param_id, default in params:
            row = tk.Frame(frame, bg=COLORS['readonly_bg'])
            row.pack(fill='x', padx=3, pady=1)

            # Get label from translation
            label = t(f'param_labels.{param_id}')
            lbl = tk.Label(row, text=label, width=18, anchor='w',
                          bg=COLORS['readonly_bg'],
                          font=('TkDefaultFont', self.font_size))
            lbl.pack(side='left')
            self.param_labels[param_id] = lbl

            var = tk.StringVar(value=str(default))
            entry = tk.Entry(row, textvariable=var, width=10, state='disabled',
                           disabledbackground=COLORS['readonly_bg'],
                           disabledforeground=COLORS['original_fg'],
                           font=('TkDefaultFont', self.font_size))
            entry.pack(side='left', padx=2)
            entry.bind('<KeyRelease>', lambda e, p=param_id: self._on_value_change(p))

            reset_btn = tk.Button(row, text="↺", width=2, state='disabled',
                                 command=lambda p=param_id: self._reset_value(p),
                                 font=('TkDefaultFont', self.font_size - 2))
            reset_btn.pack(side='left', padx=1)

            # Boutons +/- pour les paramètres en pourcentage
            if param_id in self.PERCENT_PARAMS:
                minus_btn = tk.Button(row, text="-", width=2, state='disabled',
                                     command=lambda p=param_id: self._decrement_param(p),
                                     font=('TkDefaultFont', self.font_size - 2))
                minus_btn.pack(side='left', padx=1)

                plus_btn = tk.Button(row, text="+", width=2, state='disabled',
                                    command=lambda p=param_id: self._increment_param(p),
                                    font=('TkDefaultFont', self.font_size - 2))
                plus_btn.pack(side='left', padx=1)

                self.param_plus_btns[param_id] = plus_btn
                self.param_minus_btns[param_id] = minus_btn

            self.param_vars[param_id] = var
            self.param_entries[param_id] = entry
            self.param_reset_btns[param_id] = reset_btn
            self.original_values[param_id] = str(default)

    def update_font_size(self, new_size):
        """Met à jour la taille de police de tous les widgets"""
        self.font_size = new_size
        for frame in self.section_frames:
            frame.configure(font=('TkDefaultFont', new_size, 'bold'))
        for lbl in self.param_labels.values():
            lbl.configure(font=('TkDefaultFont', new_size))
        for entry in self.param_entries.values():
            entry.configure(font=('TkDefaultFont', new_size))
        for btn in self.param_reset_btns.values():
            btn.configure(font=('TkDefaultFont', new_size - 2))
        for btn in self.param_plus_btns.values():
            btn.configure(font=('TkDefaultFont', new_size - 2))
        for btn in self.param_minus_btns.values():
            btn.configure(font=('TkDefaultFont', new_size - 2))

    def refresh_labels(self):
        """Met à jour tous les labels avec la langue courante"""
        # Update checkbox
        self.edit_check.configure(text=f"✏️ {t('panel.allow_modifications')}")
        # Update section titles
        for frame, section_key in self.section_keys.items():
            frame.configure(text=t(f'param_sections.{section_key}'))
        # Update param labels
        for param_id, lbl in self.param_labels.items():
            lbl.configure(text=t(f'param_labels.{param_id}'))

    def _on_edit_scenario_click(self):
        """Bascule entre la vue paramètres et l'éditeur de scénario"""
        if self.scenario_editor_mode:
            # Fermer l'éditeur
            self._close_scenario_editor()
        else:
            # Ouvrir l'éditeur
            self._open_scenario_editor()

    def set_scenario_file(self, file_path):
        """Définit le fichier scénario à éditer"""
        self.scenario_file_path = file_path

    def _open_scenario_editor(self):
        """Ouvre l'éditeur de scénario intégré"""
        if not self.scenario_file_path:
            return

        # Charger le contenu du fichier
        try:
            # Mettre à jour l'état de l'éditeur AVANT de charger le contenu
            self._update_scenario_editor_state()

            # Activer temporairement pour pouvoir éditer le contenu
            self.scenario_editor_text.configure(state='normal')
            content = self.scenario_file_path.read_text(encoding='utf-8')
            self.scenario_editor_text.delete('1.0', 'end')
            self.scenario_editor_text.insert('1.0', content)

            # Remettre l'état correct selon le verrouillage
            if not self.editable:
                self.scenario_editor_text.configure(state='disabled')

            # Mettre à jour le nom du fichier et le status
            self.scenario_name_label.configure(text=f"📝 {self.scenario_file_path.name}")
            self.scenario_editor_status.set("")
        except Exception as e:
            self.scenario_editor_status.set(f"Erreur: {e}")
            return

        # Sauvegarder la hauteur actuelle pour éviter le redimensionnement
        self._saved_height = self.winfo_height()
        self.configure(height=self._saved_height)
        self.pack_propagate(False)

        # Masquer le header et params_container, afficher l'éditeur
        for widget in self.winfo_children():
            if widget != self.scenario_editor_frame:
                widget.pack_forget()
        self.scenario_editor_frame.pack(fill='both', expand=True)
        self.scenario_editor_mode = True
        self.edit_scenario_btn.configure(text="📊")

    def _close_scenario_editor(self):
        """Ferme l'éditeur et retourne à la vue paramètres"""
        self.scenario_editor_frame.pack_forget()

        # Restaurer la propagation normale
        self.pack_propagate(True)

        # Réafficher les widgets dans l'ordre correct
        self.header_frame.pack(fill='x', padx=5, pady=3)
        self.params_container.pack(fill='x', padx=5, pady=3)

        self.scenario_editor_mode = False
        self.edit_scenario_btn.configure(text="✎")
        self.scenario_editor_status.set("")

    def _save_scenario_editor(self):
        """Sauvegarde le contenu de l'éditeur de scénario"""
        if not self.scenario_file_path:
            return

        content = self.scenario_editor_text.get('1.0', 'end-1c')

        # Valider la syntaxe INI
        try:
            parser = configparser.ConfigParser()
            parser.read_string(content)
        except Exception as e:
            result = messagebox.askyesno(t('dialogs.error'),
                f"Erreur de syntaxe INI:\n{e}\n\nSauvegarder quand même?")
            if not result:
                return

        try:
            self.scenario_file_path.write_text(content, encoding='utf-8')
            self.scenario_editor_status.set(f"✓ {t('editor.saved')}")

            # Recharger le scénario si callback fourni
            if self.on_scenario_reload:
                self.on_scenario_reload()
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_save', error=e))

    def _validate_scenario_editor(self):
        """Valide la syntaxe INI"""
        content = self.scenario_editor_text.get('1.0', 'end-1c')
        try:
            parser = configparser.ConfigParser()
            parser.read_string(content)
            self.scenario_editor_status.set(t('status.syntax_ok'))
        except Exception as e:
            self.scenario_editor_status.set(t('editor.error'))
            messagebox.showerror(t('dialogs.error'), str(e))

    def _toggle_edit_mode(self):
        self.editable = self.edit_var.get()
        self._update_colors()

        state = 'normal' if self.editable else 'disabled'
        for entry in self.param_entries.values():
            entry.configure(state=state)
        for btn in self.param_reset_btns.values():
            btn.configure(state=state)
        # Activer/désactiver les boutons +/-
        for btn in self.param_plus_btns.values():
            btn.configure(state=state)
        for btn in self.param_minus_btns.values():
            btn.configure(state=state)
        # Activer/désactiver le champ devise
        self.devise_entry.configure(state=state)

        # Mettre à jour l'icône du cadenas et l'état de l'éditeur texte
        self._update_scenario_editor_state()

    def _on_scenario_lock_toggle(self):
        """Appelé quand on clique sur le cadenas de l'éditeur de scénario"""
        self._toggle_edit_mode()

    def _update_scenario_editor_state(self):
        """Met à jour l'état de l'éditeur de scénario selon le verrouillage"""
        # Mettre à jour l'icône du cadenas (panneau paramètres ET éditeur texte)
        lock_icon = "🔓" if self.editable else "🔒"
        self.edit_check.configure(text=lock_icon)
        self.scenario_lock_check.configure(text=lock_icon)

        # Mettre à jour l'état et la couleur de l'éditeur texte
        if self.editable:
            self.scenario_editor_text.configure(
                state='normal',
                background='white',
                foreground='black'
            )
        else:
            self.scenario_editor_text.configure(
                state='disabled',
                background=COLORS['readonly_bg'],
                foreground='#666666'
            )

        # Mettre à jour l'état des boutons
        state = 'normal' if self.editable else 'disabled'
        self.save_scenario_btn.configure(state=state)
        self.validate_scenario_btn.configure(state=state)

    def _on_devise_change(self, event=None):
        """Appelé quand la devise change"""
        if self.on_param_change:
            self.on_param_change()

    def set_devise(self, devise):
        """Définit la devise affichée"""
        self.devise_var.set(devise)

    def get_devise(self):
        """Retourne la devise actuelle"""
        return self.devise_var.get()

    def _update_colors(self):
        bg_color = COLORS['editable_bg'] if self.editable else COLORS['readonly_bg']

        self.params_container.configure(bg=bg_color)

        # Mettre à jour les LabelFrames de section
        for frame in self.section_frames:
            frame.configure(bg=bg_color)
            for child in frame.winfo_children():
                if isinstance(child, tk.Frame):
                    child.configure(bg=bg_color)

        # Mettre à jour les labels
        for lbl in self.param_labels.values():
            lbl.configure(bg=bg_color)

        # Mettre à jour les couleurs des entrées
        for param_id, entry in self.param_entries.items():
            current_val = self.param_vars[param_id].get()
            original_val = self.original_values.get(param_id, '')
            is_modified = current_val != original_val

            if self.editable:
                fg_color = COLORS['modified_fg'] if is_modified else COLORS['original_fg']
                entry.configure(bg='white', fg=fg_color)
            else:
                entry.configure(disabledbackground=bg_color,
                              disabledforeground=COLORS['modified_fg'] if is_modified else COLORS['original_fg'])

    def _on_value_change(self, param_id):
        self._update_entry_color(param_id)
        self._debounce_refresh()  # Debounce pour permettre de taper plusieurs chiffres

    def _update_entry_color(self, param_id):
        if param_id not in self.param_vars:
            return

        current_val = self.param_vars[param_id].get()
        original_val = self.original_values.get(param_id, '')
        is_modified = current_val != original_val

        entry = self.param_entries[param_id]
        fg_color = COLORS['modified_fg'] if is_modified else COLORS['original_fg']
        entry.configure(fg=fg_color)

    def _reset_value(self, param_id):
        if param_id in self.original_values:
            self.param_vars[param_id].set(self.original_values[param_id])
            self._update_entry_color(param_id)
            if self.on_param_change:
                self.on_param_change()

    def _increment_param(self, param_id):
        """Incrémente un paramètre pourcentage (avec debounce)"""
        if param_id not in self.PERCENT_PARAMS:
            return
        increment = self.PERCENT_PARAMS[param_id]
        try:
            current = float(self.param_vars[param_id].get())
            new_value = current + increment
            self.param_vars[param_id].set(f"{new_value:.2f}".rstrip('0').rstrip('.'))
            self._update_entry_color(param_id)
            self._debounce_refresh()
        except ValueError:
            pass

    def _decrement_param(self, param_id):
        """Décrémente un paramètre pourcentage (avec debounce)"""
        if param_id not in self.PERCENT_PARAMS:
            return
        increment = self.PERCENT_PARAMS[param_id]
        try:
            current = float(self.param_vars[param_id].get())
            new_value = max(0, current - increment)  # Ne pas descendre sous 0
            self.param_vars[param_id].set(f"{new_value:.2f}".rstrip('0').rstrip('.'))
            self._update_entry_color(param_id)
            self._debounce_refresh()
        except ValueError:
            pass

    def _debounce_refresh(self):
        """Rafraîchit les graphiques après 500ms d'inactivité"""
        # Annuler le timer précédent s'il existe
        if self._debounce_timer is not None:
            self.after_cancel(self._debounce_timer)
        # Créer un nouveau timer
        self._debounce_timer = self.after(500, self._do_refresh)

    def _do_refresh(self):
        """Effectue le rafraîchissement des graphiques"""
        self._debounce_timer = None
        if self.on_param_change:
            self.on_param_change()

    def get_modified_params_dict(self):
        """Retourne un dict des paramètres modifiés: {param_name: new_value}"""
        modified = {}
        for param_id, var in self.param_vars.items():
            current_val = var.get()
            original_val = self.original_values.get(param_id, '')
            if current_val != original_val:
                try:
                    # Convertir en float
                    value = float(current_val)
                    modified[param_id] = value
                except ValueError:
                    pass
        return modified

    def load_params(self, params: Parametres):
        # Remettre en mode lecture seule quand on charge un nouveau scénario
        self.edit_var.set(False)
        self.editable = False
        self._toggle_edit_mode()

        mappings = {
            'pib_initial': params.pib_initial,
            'dette_publique_initiale': params.dette_publique_initiale,
            'taux_croissance_base': params.taux_croissance_base * 100,
            'dette_implicite_initiale': params.dette_implicite_initiale,
            'reduction_solidaire': params.reduction_solidaire * 100,
            'pension_moyenne_annuelle': params.pension_moyenne_annuelle,
            'privatisations_totales': params.privatisations_totales,
            'differentiel_initial': params.differentiel_initial * 100,
            'duree_decroissance_differentiel': params.duree_decroissance_differentiel,
            'taux_flat_tax': params.taux_flat_tax * 100,
            'abattement_forfaitaire': params.abattement_forfaitaire,
            'assurance_sante': params.assurance_sante,
            'assurance_chomage': params.assurance_chomage,
            'assurance_pension': params.assurance_pension,
            'assurance_education': params.assurance_education,
            'surplus_budgetaire_minimal_pct_pib': params.surplus_budgetaire_minimal_pct_pib * 100,
            'surplus_max_pour_dette_transition_pct': params.surplus_max_pour_dette_transition_pct * 100,
        }

        for param_id, value in mappings.items():
            if param_id in self.param_vars:
                str_val = str(value)
                self.param_vars[param_id].set(str_val)
                self.original_values[param_id] = str_val

        self._update_colors()

    def get_modified_params(self, base_params: Parametres) -> Parametres:
        params = Parametres(
            nom_pays=base_params.nom_pays,
            nombre_travailleurs_actifs=base_params.nombre_travailleurs_actifs,
            bonus_croissance_1_10=base_params.bonus_croissance_1_10,
            bonus_croissance_11_20=base_params.bonus_croissance_11_20,
            bonus_croissance_20_plus=base_params.bonus_croissance_20_plus,
            nombre_retraites_initiaux=base_params.nombre_retraites_initiaux,
            nouveaux_retraites_par_an=base_params.nouveaux_retraites_par_an,
            duree_carriere=base_params.duree_carriere,
            age_depart_retraite=base_params.age_depart_retraite,
            age_esperance_vie=base_params.age_esperance_vie,
            taux_mortalite_initial=base_params.taux_mortalite_initial,
            increment_mortalite_annuel=base_params.increment_mortalite_annuel,
            methode_droits=base_params.methode_droits,
            parametre_methode_droits=base_params.parametre_methode_droits,
            methode_differentiel=base_params.methode_differentiel,
            parametre_methode_differentiel=base_params.parametre_methode_differentiel,
            remboursement_dette_pub_pct_pib=base_params.remboursement_dette_pub_pct_pib,
            seuils_taux_interet=base_params.seuils_taux_interet,
            salaires_analyses=base_params.salaires_analyses,
            prelevement_actuel_total=base_params.prelevement_actuel_total,
            taxes_indirectes=base_params.taxes_indirectes,
        )

        try:
            params.pib_initial = float(self.param_vars['pib_initial'].get())
            params.dette_publique_initiale = float(self.param_vars['dette_publique_initiale'].get())
            params.taux_croissance_base = float(self.param_vars['taux_croissance_base'].get()) / 100
            params.dette_implicite_initiale = float(self.param_vars['dette_implicite_initiale'].get())
            params.reduction_solidaire = float(self.param_vars['reduction_solidaire'].get()) / 100
            params.pension_moyenne_annuelle = float(self.param_vars['pension_moyenne_annuelle'].get())
            params.privatisations_totales = float(self.param_vars['privatisations_totales'].get())
            params.differentiel_initial = float(self.param_vars['differentiel_initial'].get()) / 100
            params.duree_decroissance_differentiel = int(float(self.param_vars['duree_decroissance_differentiel'].get()))
            params.taux_flat_tax = float(self.param_vars['taux_flat_tax'].get()) / 100
            params.abattement_forfaitaire = float(self.param_vars['abattement_forfaitaire'].get())
            params.assurance_sante = float(self.param_vars['assurance_sante'].get())
            params.assurance_chomage = float(self.param_vars['assurance_chomage'].get())
            params.assurance_pension = float(self.param_vars['assurance_pension'].get())
            params.assurance_education = float(self.param_vars['assurance_education'].get())
            params.surplus_budgetaire_minimal_pct_pib = float(self.param_vars['surplus_budgetaire_minimal_pct_pib'].get()) / 100
            params.surplus_max_pour_dette_transition_pct = float(self.param_vars['surplus_max_pour_dette_transition_pct'].get()) / 100
        except ValueError:
            pass

        return params


class VisualGraphEditor(tk.Toplevel):
    """Éditeur visuel de graphiques avec formulaires, menus déroulants et sélecteur de couleur"""

    # Field codes (keys) - labels from translations
    FIELD_CODES = ['ANNE', 'PIBS', 'DIFF', 'INTR', 'DPUB', 'DTRA', 'DTOT', 'DPPC', 'DTPC', 'TTPC', 'DIMP', 'DIPC', 'FLUX', 'NRET']

    # Graph type codes
    GRAPH_TYPE_CODES = ['line', 'bar', 'stacked_bar']

    # Line style codes
    LINE_STYLE_CODES = ['solid', 'dashed', 'dotted']

    # Color presets (not translated)
    COLORS_PRESETS = [
        '#4A90E2', '#93c5fd', '#16a34a', '#86efac', '#f97316', '#fdba74',
        '#ef4444', '#fca5a5', '#8b5cf6', '#c4b5fd', '#374151', '#d1d5db',
    ]

    def _get_field_options(self):
        """Get field options with translated labels"""
        return [(code, t(f'fields.{code}')) for code in self.FIELD_CODES]

    def _get_graph_types(self):
        """Get graph types with translated labels"""
        return [(code, t(f'graph_types.{code}')) for code in self.GRAPH_TYPE_CODES]

    def _get_line_styles(self):
        """Get line styles with translated labels"""
        return [(code, t(f'line_styles.{code}')) for code in self.LINE_STYLE_CODES]

    def __init__(self, parent, file_path, font_size=BASE_FONT_SIZE, on_save=None):
        super().__init__(parent)
        self.file_path = Path(file_path)
        self.font_size = font_size
        self.on_save_callback = on_save
        self.graph_data = {}
        self.curve_widgets = []

        self.title(f"{t('editor.visual_title')} - {self.file_path.name}")
        self.geometry("700x800")

        self._create_widgets()
        self._load_file()

    def _create_widgets(self):
        # === TOOLBAR ===
        toolbar = ttk.Frame(self)
        toolbar.pack(fill='x', padx=5, pady=5)

        ttk.Button(toolbar, text=f"💾 {t('editor.save')}", command=self._save_file).pack(side='left', padx=2)
        ttk.Button(toolbar, text=f"📝 {t('editor.text_mode')}", command=self._switch_to_text_mode).pack(side='left', padx=2)

        self.status_var = tk.StringVar(value="")
        ttk.Label(toolbar, textvariable=self.status_var).pack(side='right', padx=10)

        # === CONTENU SCROLLABLE ===
        canvas = tk.Canvas(self)
        scrollbar = ttk.Scrollbar(self, orient="vertical", command=canvas.yview)
        self.scrollable_frame = ttk.Frame(canvas)

        self.scrollable_frame.bind(
            "<Configure>",
            lambda e: canvas.configure(scrollregion=canvas.bbox("all"))
        )

        canvas.create_window((0, 0), window=self.scrollable_frame, anchor="nw")
        canvas.configure(yscrollcommand=scrollbar.set)

        scrollbar.pack(side="right", fill="y")
        canvas.pack(side="left", fill="both", expand=True)

        # Bind mouse wheel
        canvas.bind_all("<MouseWheel>", lambda e: canvas.yview_scroll(int(-1*(e.delta/120)), "units"))

        # === PROPRIÉTÉS GÉNÉRALES ===
        general_frame = ttk.LabelFrame(self.scrollable_frame, text=f"📊 {t('editor.general_props')}")
        general_frame.pack(fill='x', padx=10, pady=5)

        # Titre
        ttk.Label(general_frame, text=t('editor.graph_title')).grid(row=0, column=0, sticky='w', padx=5, pady=3)
        self.title_var = tk.StringVar()
        ttk.Entry(general_frame, textvariable=self.title_var, width=50).grid(row=0, column=1, columnspan=2, sticky='w', padx=5, pady=3)

        # Type de graphique
        ttk.Label(general_frame, text=t('editor.graph_type')).grid(row=1, column=0, sticky='w', padx=5, pady=3)
        self.type_var = tk.StringVar()
        type_combo = ttk.Combobox(general_frame, textvariable=self.type_var, state='readonly', width=20)
        type_combo['values'] = [f"{code} - {name}" for code, name in self._get_graph_types()]
        type_combo.grid(row=1, column=1, sticky='w', padx=5, pady=3)

        # === AXE X ===
        x_frame = ttk.LabelFrame(self.scrollable_frame, text=f"↔️ {t('editor.x_axis')}")
        x_frame.pack(fill='x', padx=10, pady=5)

        ttk.Label(x_frame, text=t('editor.field')).grid(row=0, column=0, sticky='w', padx=5, pady=3)
        self.x_field_var = tk.StringVar()
        x_field_combo = ttk.Combobox(x_frame, textvariable=self.x_field_var, state='readonly', width=30)
        x_field_combo['values'] = [f"{code} - {name}" for code, name in self._get_field_options()]
        x_field_combo.grid(row=0, column=1, sticky='w', padx=5, pady=3)

        ttk.Label(x_frame, text=t('editor.label')).grid(row=1, column=0, sticky='w', padx=5, pady=3)
        self.x_label_var = tk.StringVar()
        ttk.Entry(x_frame, textvariable=self.x_label_var, width=40).grid(row=1, column=1, sticky='w', padx=5, pady=3)

        # === AXE Y ===
        y_frame = ttk.LabelFrame(self.scrollable_frame, text=f"↕️ {t('editor.y_axis')}")
        y_frame.pack(fill='x', padx=10, pady=5)

        ttk.Label(y_frame, text=t('editor.label')).grid(row=0, column=0, sticky='w', padx=5, pady=3)
        self.y_label_var = tk.StringVar()
        ttk.Entry(y_frame, textvariable=self.y_label_var, width=40).grid(row=0, column=1, sticky='w', padx=5, pady=3)

        ttk.Label(y_frame, text=t('editor.divisor')).grid(row=1, column=0, sticky='w', padx=5, pady=3)
        self.y_divisor_var = tk.StringVar(value="1")
        ttk.Entry(y_frame, textvariable=self.y_divisor_var, width=10).grid(row=1, column=1, sticky='w', padx=5, pady=3)

        ttk.Label(y_frame, text=t('editor.min')).grid(row=2, column=0, sticky='w', padx=5, pady=3)
        self.y_min_var = tk.StringVar()
        ttk.Entry(y_frame, textvariable=self.y_min_var, width=10).grid(row=2, column=1, sticky='w', padx=5, pady=3)

        # === COURBES ===
        self.curves_frame = ttk.LabelFrame(self.scrollable_frame, text=f"📈 {t('editor.curves')}")
        self.curves_frame.pack(fill='x', padx=10, pady=5)

        ttk.Button(self.curves_frame, text=f"➕ {t('editor.add_curve')}", command=self._add_curve_widget).pack(pady=5)

        self.curves_container = ttk.Frame(self.curves_frame)
        self.curves_container.pack(fill='x', padx=5, pady=5)

    def _add_curve_widget(self, data=None):
        """Ajoute un widget pour une courbe"""
        idx = len(self.curve_widgets)
        frame = ttk.LabelFrame(self.curves_container, text=t('editor.curve_n', n=idx + 1))
        frame.pack(fill='x', pady=3)

        widgets = {}

        # Ligne 1: Champ + Label
        ttk.Label(frame, text=t('editor.field')).grid(row=0, column=0, sticky='w', padx=3, pady=2)
        widgets['field'] = tk.StringVar(value=data.get('field', '') if data else '')
        field_combo = ttk.Combobox(frame, textvariable=widgets['field'], state='readonly', width=25)
        field_combo['values'] = [f"{code} - {name}" for code, name in self._get_field_options()]
        field_combo.grid(row=0, column=1, sticky='w', padx=3, pady=2)

        ttk.Label(frame, text=t('editor.label')).grid(row=0, column=2, sticky='w', padx=3, pady=2)
        widgets['label'] = tk.StringVar(value=data.get('label', '') if data else '')
        ttk.Entry(frame, textvariable=widgets['label'], width=20).grid(row=0, column=3, sticky='w', padx=3, pady=2)

        # Ligne 2: Couleur + Style
        ttk.Label(frame, text=t('editor.color')).grid(row=1, column=0, sticky='w', padx=3, pady=2)
        widgets['color'] = tk.StringVar(value=data.get('color', '#4A90E2') if data else '#4A90E2')

        color_frame = ttk.Frame(frame)
        color_frame.grid(row=1, column=1, sticky='w', padx=3, pady=2)

        color_combo = ttk.Combobox(color_frame, textvariable=widgets['color'], width=15)
        color_combo['values'] = self.COLORS_PRESETS
        color_combo.pack(side='left')

        # Bouton sélecteur de couleur
        widgets['color_btn'] = tk.Button(color_frame, text="🎨", width=3,
                                         command=lambda: self._pick_color(widgets['color']))
        widgets['color_btn'].pack(side='left', padx=2)

        ttk.Label(frame, text=t('editor.style')).grid(row=1, column=2, sticky='w', padx=3, pady=2)
        widgets['style'] = tk.StringVar(value=data.get('style', 'solid') if data else 'solid')
        style_combo = ttk.Combobox(frame, textvariable=widgets['style'], state='readonly', width=15)
        style_combo['values'] = [code for code, name in self._get_line_styles()]
        style_combo.grid(row=1, column=3, sticky='w', padx=3, pady=2)

        # Ligne 3: Fill + Alpha
        widgets['fill'] = tk.BooleanVar(value=data.get('fill', False) if data else False)
        ttk.Checkbutton(frame, text=t('editor.fill'), variable=widgets['fill']).grid(row=2, column=0, columnspan=2, sticky='w', padx=3, pady=2)

        ttk.Label(frame, text=t('editor.alpha')).grid(row=2, column=2, sticky='w', padx=3, pady=2)
        widgets['fill_alpha'] = tk.StringVar(value=str(data.get('fill_alpha', 0.2)) if data else '0.2')
        ttk.Entry(frame, textvariable=widgets['fill_alpha'], width=8).grid(row=2, column=3, sticky='w', padx=3, pady=2)

        # Bouton supprimer
        ttk.Button(frame, text="🗑️", width=3,
                   command=lambda f=frame, i=idx: self._remove_curve(f, i)).grid(row=0, column=4, rowspan=3, padx=5)

        widgets['frame'] = frame
        self.curve_widgets.append(widgets)

    def _remove_curve(self, frame, idx):
        """Supprime une courbe"""
        frame.destroy()
        self.curve_widgets = [w for w in self.curve_widgets if w.get('frame') != frame]
        # Renuméroter les courbes
        for i, w in enumerate(self.curve_widgets):
            w['frame'].configure(text=f"Courbe {i + 1}")

    def _pick_color(self, color_var):
        """Ouvre le sélecteur de couleur"""
        from tkinter import colorchooser
        color = colorchooser.askcolor(color=color_var.get(), title="Choisir une couleur")
        if color[1]:
            color_var.set(color[1])

    def _load_file(self):
        """Charge le fichier YAML et remplit les widgets"""
        try:
            content = self.file_path.read_text(encoding='utf-8')
            self.graph_data = yaml.safe_load(content) or {}
            self._populate_widgets()
            self.status_var.set(f"✓ {t('editor.file_loaded')}")
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_load', error=e))

    def _populate_widgets(self):
        """Remplit les widgets avec les données du YAML"""
        d = self.graph_data

        # Titre
        self.title_var.set(get_yaml_value(d, 'title', 'titre', ''))

        # Type
        graph_type = d.get('type', 'line')
        for code, name in self._get_graph_types():
            if code == graph_type:
                self.type_var.set(f"{code} - {name}")
                break

        # Axe X
        x_axis = get_yaml_value(d, 'x_axis', 'axe_x', {})
        x_field = get_yaml_value(x_axis, 'field', 'champ', '')
        for code, name in self._get_field_options():
            if code == x_field:
                self.x_field_var.set(f"{code} - {name}")
                break
        self.x_label_var.set(get_yaml_value(x_axis, 'label', 'label', ''))

        # Axe Y
        y_axis = get_yaml_value(d, 'y_axis', 'axe_y', {})
        self.y_label_var.set(get_yaml_value(y_axis, 'label', 'label', ''))
        self.y_divisor_var.set(str(get_yaml_value(y_axis, 'divisor', 'diviseur', 1)))
        y_min = y_axis.get('min')
        self.y_min_var.set(str(y_min) if y_min is not None else '')

        # Courbes
        curves = get_yaml_value(d, 'curves', 'courbes', [])
        for curve in curves:
            field_code = get_yaml_value(curve, 'field', 'champ', '')
            # Trouver le bon format pour le combobox
            for code, name in self._get_field_options():
                if code == field_code:
                    curve['field'] = f"{code} - {name}"
                    break
            self._add_curve_widget(curve)

    def _save_file(self):
        """Sauvegarde les données dans le fichier YAML"""
        try:
            # Construire le dictionnaire YAML
            data = {
                'title': self.title_var.get(),
                'type': self.type_var.get().split(' - ')[0] if ' - ' in self.type_var.get() else self.type_var.get(),
            }

            # Axe X
            x_field = self.x_field_var.get().split(' - ')[0] if ' - ' in self.x_field_var.get() else self.x_field_var.get()
            data['x_axis'] = {
                'field': x_field,
                'label': self.x_label_var.get(),
            }

            # Axe Y
            data['y_axis'] = {
                'label': self.y_label_var.get(),
            }
            try:
                data['y_axis']['divisor'] = int(self.y_divisor_var.get()) if self.y_divisor_var.get() else 1
            except ValueError:
                data['y_axis']['divisor'] = float(self.y_divisor_var.get()) if self.y_divisor_var.get() else 1

            if self.y_min_var.get():
                try:
                    data['y_axis']['min'] = int(self.y_min_var.get())
                except ValueError:
                    data['y_axis']['min'] = float(self.y_min_var.get())

            # Courbes
            data['curves'] = []
            for w in self.curve_widgets:
                field = w['field'].get().split(' - ')[0] if ' - ' in w['field'].get() else w['field'].get()
                curve = {
                    'field': field,
                    'label': w['label'].get(),
                    'color': w['color'].get(),
                    'style': w['style'].get(),
                }
                if w['fill'].get():
                    curve['fill'] = True
                    try:
                        curve['fill_alpha'] = float(w['fill_alpha'].get())
                    except ValueError:
                        curve['fill_alpha'] = 0.2
                data['curves'].append(curve)

            # Sauvegarder
            yaml_content = yaml.dump(data, allow_unicode=True, default_flow_style=False, sort_keys=False)
            self.file_path.write_text(yaml_content, encoding='utf-8')
            self.status_var.set(f"✓ {t('editor.saved')}")

            if self.on_save_callback:
                self.on_save_callback()

        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_save', error=e))

    def _switch_to_text_mode(self):
        """Ouvre l'éditeur texte classique"""
        self.destroy()
        EditorWindow(self.master, self.file_path, 'yaml', self.font_size, self.on_save_callback)


class EditorWindow(tk.Toplevel):
    """Fenêtre d'édition de fichiers YAML/INI"""

    def __init__(self, parent, file_path, file_type='yaml', font_size=BASE_FONT_SIZE, on_save=None):
        super().__init__(parent)
        self.file_path = Path(file_path)
        self.file_type = file_type
        self.font_size = font_size
        self.on_save_callback = on_save
        self.title(f"{t('editor.title')} - {self.file_path.name}")
        self.geometry("900x700")

        self._create_widgets()
        self._load_file()

    def _create_widgets(self):
        toolbar = ttk.Frame(self)
        toolbar.pack(fill='x', padx=5, pady=5)

        ttk.Button(toolbar, text=f"💾 {t('editor.save')}", command=self._save_file).pack(side='left', padx=2)
        ttk.Button(toolbar, text=f"✓ {t('editor.validate')}", command=self._validate_syntax).pack(side='left', padx=2)
        ttk.Button(toolbar, text=f"↻ {t('editor.reload')}", command=self._load_file).pack(side='left', padx=2)

        self.status_var = tk.StringVar(value="")
        ttk.Label(toolbar, textvariable=self.status_var).pack(side='right', padx=10)

        self.text = scrolledtext.ScrolledText(self, wrap='none', font=('Consolas', self.font_size))
        self.text.pack(fill='both', expand=True, padx=5, pady=5)

        h_scroll = ttk.Scrollbar(self, orient='horizontal', command=self.text.xview)
        h_scroll.pack(fill='x')
        self.text.configure(xscrollcommand=h_scroll.set)

        # Binding Select All (Ctrl+A)
        self.text.bind('<Control-a>', self._select_all_text)
        self.text.bind('<Control-A>', self._select_all_text)

    def _select_all_text(self, event):
        """Sélectionne tout le texte"""
        self.text.tag_add('sel', '1.0', 'end')
        self.text.mark_set('insert', 'end')
        return 'break'

    def update_font_size(self, new_size):
        """Met à jour la taille de police de l'éditeur"""
        self.font_size = new_size
        self.text.configure(font=('Consolas', new_size))

    def _load_file(self):
        try:
            content = self.file_path.read_text(encoding='utf-8')
            self.text.delete('1.0', 'end')
            self.text.insert('1.0', content)
            self.status_var.set(t('editor.file_loaded'))
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_load', error=e))

    def _save_file(self):
        content = self.text.get('1.0', 'end-1c')
        is_valid, error = self._do_validate(content)
        if not is_valid:
            result = messagebox.askyesno(t('dialogs.error'), t('dialogs.yaml_syntax_error', error=error))
            if not result:
                return

        try:
            self.file_path.write_text(content, encoding='utf-8')
            self.status_var.set(f"✓ {t('editor.saved')}")
            if self.on_save_callback:
                self.on_save_callback()
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_save', error=e))

    def _validate_syntax(self):
        content = self.text.get('1.0', 'end-1c')
        is_valid, error = self._do_validate(content)
        if is_valid:
            self.status_var.set(t('status.syntax_ok'))
        else:
            self.status_var.set(t('editor.error'))
            messagebox.showerror(t('dialogs.error'), error)

    def _do_validate(self, content):
        try:
            if self.file_type == 'yaml':
                yaml.safe_load(content)
            elif self.file_type == 'ini':
                parser = configparser.ConfigParser()
                parser.read_string(content)
            return True, None
        except Exception as e:
            return False, str(e)


class EnlargedGraphWindow(tk.Toplevel):
    """Fenêtre de graphique agrandi avec option Live et zoom/pan image"""

    def __init__(self, parent, graph_def, resultats, params, devise, font_size=BASE_FONT_SIZE,
                 on_update_callback=None, get_modified_params_callback=None, get_scenario_name_callback=None,
                 graph_id=None):
        super().__init__(parent)
        self.title(t('enlarged_window.title'))
        self.geometry("1200x850")

        self.graph_def = graph_def
        self.graph_id = graph_id
        self.resultats = resultats
        self.params = params
        self.devise = devise
        self.font_size = font_size
        self.on_update_callback = on_update_callback
        self.get_modified_params_callback = get_modified_params_callback
        self.get_scenario_name_callback = get_scenario_name_callback
        self.is_live = False

        # Image et état zoom/pan
        self._graph_image = None
        self._photo_image = None
        self._zoom_level = 1.0
        self._pan_offset_x = 0
        self._pan_offset_y = 0
        self._drag_start_x = 0
        self._drag_start_y = 0

        # Figure matplotlib (pour générer l'image)
        self.fig = Figure(figsize=(12, 8), dpi=100)
        self.ax = self.fig.add_subplot(111)

        self._status_clear_job = None
        self._create_widgets()

    def _set_status(self, message, timeout=5000, error=False):
        """Met à jour la barre de statut avec un message temporaire"""
        if self._status_clear_job:
            self.after_cancel(self._status_clear_job)
        self.status_var.set(message)
        self.status_label.configure(fg='#dc2626' if error else 'black')
        def clear_status():
            self.status_var.set(t('status.ready'))
            self.status_label.configure(fg='black')
        self._status_clear_job = self.after(timeout, clear_status)

    def _add_tooltip(self, widget, text):
        """Ajoute un tooltip à un widget"""
        def show_tooltip(event):
            tooltip = tk.Toplevel(widget)
            tooltip.wm_overrideredirect(True)
            tooltip.wm_geometry(f"+{event.x_root+10}+{event.y_root+10}")
            label = tk.Label(tooltip, text=text, background="#ffffe0", relief='solid', borderwidth=1,
                           font=('TkDefaultFont', self.font_size - 2))
            label.pack()
            widget._tooltip = tooltip
            widget.after(2000, lambda: tooltip.destroy() if hasattr(widget, '_tooltip') else None)

        def hide_tooltip(event):
            if hasattr(widget, '_tooltip') and widget._tooltip:
                widget._tooltip.destroy()
                widget._tooltip = None

        widget.bind('<Enter>', show_tooltip)
        widget.bind('<Leave>', hide_tooltip)

    def _create_widgets(self):
        # Barre de statut en bas
        status_frame = ttk.Frame(self)
        status_frame.pack(side='bottom', fill='x')
        self.status_var = tk.StringVar(value=t('status.ready'))
        self.status_label = tk.Label(status_frame, textvariable=self.status_var,
                                     relief='sunken', anchor='w', padx=5, pady=2,
                                     bg='#f0f0f0', fg='black')
        self.status_label.pack(fill='x')

        # Toolbar avec checkbox Live et bouton save
        toolbar = ttk.Frame(self)
        toolbar.pack(fill='x', padx=5, pady=5)

        self.live_var = tk.BooleanVar(value=False)
        live_check = ttk.Checkbutton(
            toolbar,
            text=f"🔴 {t('enlarged_window.live')}",
            variable=self.live_var,
            command=self._toggle_live
        )
        live_check.pack(side='left')

        # Instructions zoom/pan
        ttk.Label(toolbar, text="🔍 Molette=zoom | Drag=pan | Double-clic droit=reset",
                  font=('TkDefaultFont', 9)).pack(side='left', padx=20)

        # Boutons icônes avec tooltips (comme GraphPanel)
        save_btn = ttk.Button(toolbar, text="💾", width=3, command=self._save_graph)
        save_btn.pack(side='right', padx=2)
        self._add_tooltip(save_btn, t('tooltip.save_graph'))

        copy_png_btn = ttk.Button(toolbar, text="📋", width=3, command=self._copy_graph_png)
        copy_png_btn.pack(side='right', padx=2)
        self._add_tooltip(copy_png_btn, t('tooltip.copy_png'))

        copy_svg_btn = ttk.Button(toolbar, text="📄", width=3, command=self._copy_graph_svg)
        copy_svg_btn.pack(side='right', padx=2)
        self._add_tooltip(copy_svg_btn, t('tooltip.copy_svg'))

        reset_zoom_btn = ttk.Button(toolbar, text="⟲", width=3, command=self._reset_zoom)
        reset_zoom_btn.pack(side='right', padx=2)
        self._add_tooltip(reset_zoom_btn, t('tooltip.reset_zoom'))

        # Canvas tkinter pour afficher l'image avec zoom/pan
        self.display_canvas = tk.Canvas(self, bg='white', highlightthickness=0)
        self.display_canvas.pack(fill='both', expand=True, padx=10, pady=10)

        # Bind pour zoom avec molette (Ctrl=graphique, sans Ctrl=image)
        self.display_canvas.bind('<MouseWheel>', self._on_mousewheel)
        self.display_canvas.bind('<Button-4>', self._on_scroll_up)   # Linux scroll up
        self.display_canvas.bind('<Button-5>', self._on_scroll_down) # Linux scroll down

        # Bind pour pan avec drag
        self.display_canvas.bind('<ButtonPress-1>', self._on_drag_start)
        self.display_canvas.bind('<B1-Motion>', self._on_drag_motion)

        # Double-clic droit pour reset zoom/pan
        self.display_canvas.bind('<Double-Button-3>', lambda e: self._reset_zoom())

        # Redimensionnement auto
        self.display_canvas.bind('<Configure>', self._on_canvas_resize)

        # Échap pour fermer la fenêtre
        self.bind('<Escape>', lambda e: self.destroy())

        # Dessiner le graphique après un délai pour laisser le canvas s'initialiser
        self.after(100, self._draw_graph)

    def _on_scroll_up(self, event):
        """Linux scroll up - Ctrl=zoom graphique in, sans Ctrl=zoom image in"""
        if event.state & 0x4:
            self._zoom_graph(0.85)
        else:
            self._zoom_image(1.15)

    def _on_scroll_down(self, event):
        """Linux scroll down - Ctrl=zoom graphique out, sans Ctrl=zoom image out"""
        if event.state & 0x4:
            self._zoom_graph(1.18)
        else:
            self._zoom_image(0.87)

    def _on_mousewheel(self, event):
        """Handle mouse wheel - Ctrl=zoom graphique, sans Ctrl=zoom image"""
        ctrl_pressed = event.state & 0x4
        if ctrl_pressed:
            if event.delta > 0:
                self._zoom_graph(0.85)
            else:
                self._zoom_graph(1.18)
        else:
            if event.delta > 0:
                self._zoom_image(1.15)
            else:
                self._zoom_image(0.87)

    def _zoom_image(self, factor):
        """Zoom sur l'image (comme une loupe)"""
        if not self._graph_image:
            return

        new_zoom = self._zoom_level * factor
        if new_zoom < 0.5 or new_zoom > 4.0:
            return

        self._zoom_level = new_zoom
        self._update_display()

    def _zoom_graph(self, factor):
        """Zoom sur les données du graphique (modifie les limites des axes)"""
        if not hasattr(self, '_original_xlim') or not self._original_xlim:
            return

        xlim = self.ax.get_xlim()
        ylim = self.ax.get_ylim()

        x_center = (xlim[0] + xlim[1]) / 2
        y_center = (ylim[0] + ylim[1]) / 2

        x_half = (xlim[1] - xlim[0]) / 2 * factor
        y_half = (ylim[1] - ylim[0]) / 2 * factor

        orig_x_range = self._original_xlim[1] - self._original_xlim[0]
        orig_y_range = self._original_ylim[1] - self._original_ylim[0]
        if x_half * 2 < orig_x_range * 0.1 or x_half * 2 > orig_x_range * 3:
            return
        if y_half * 2 < orig_y_range * 0.1 or y_half * 2 > orig_y_range * 3:
            return

        self.ax.set_xlim(x_center - x_half, x_center + x_half)
        self.ax.set_ylim(y_center - y_half, y_center + y_half)

        self._render_to_image(reset_view=False)

    def _reset_zoom(self):
        """Remet le zoom image ET les axes à leur état original"""
        self._zoom_level = 1.0
        self._pan_offset_x = 0
        self._pan_offset_y = 0
        if hasattr(self, '_original_xlim') and self._original_xlim:
            self.ax.set_xlim(self._original_xlim)
            self.ax.set_ylim(self._original_ylim)
            self._render_to_image(reset_view=False)
        else:
            self._update_display()

    def _on_drag_start(self, event):
        """Début du drag pour pan"""
        self._drag_start_x = event.x
        self._drag_start_y = event.y
        self._drag_ctrl = event.state & 0x4
        if self._drag_ctrl:
            self._drag_xlim = self.ax.get_xlim()
            self._drag_ylim = self.ax.get_ylim()

    def _on_drag_motion(self, event):
        """Mouvement pendant le drag - Ctrl=pan graphique, sans Ctrl=pan image"""
        if self._drag_ctrl:
            if not hasattr(self, '_original_xlim') or not self._original_xlim:
                return
            if not hasattr(self, '_drag_xlim'):
                return

            dx_pixels = event.x - self._drag_start_x
            dy_pixels = event.y - self._drag_start_y

            canvas_width = self.display_canvas.winfo_width()
            canvas_height = self.display_canvas.winfo_height()

            x_range = self._drag_xlim[1] - self._drag_xlim[0]
            y_range = self._drag_ylim[1] - self._drag_ylim[0]

            dx_data = -dx_pixels / canvas_width * x_range
            dy_data = dy_pixels / canvas_height * y_range

            new_xlim = (self._drag_xlim[0] + dx_data, self._drag_xlim[1] + dx_data)
            new_ylim = (self._drag_ylim[0] + dy_data, self._drag_ylim[1] + dy_data)

            self.ax.set_xlim(new_xlim)
            self.ax.set_ylim(new_ylim)

            self._render_to_image(reset_view=False)
        else:
            if not self._graph_image:
                return

            dx = event.x - self._drag_start_x
            dy = event.y - self._drag_start_y

            self._pan_offset_x += dx
            self._pan_offset_y += dy

            self._drag_start_x = event.x
            self._drag_start_y = event.y

            self._update_display()

    def _on_canvas_resize(self, event):
        """Redessine quand la fenêtre change de taille"""
        # Redessiner le graphique pour qu'il remplisse la nouvelle taille
        self._draw_graph()

    def _update_display(self):
        """Met à jour l'affichage de l'image avec zoom et pan"""
        if not self._graph_image:
            return

        canvas_width = self.display_canvas.winfo_width()
        canvas_height = self.display_canvas.winfo_height()

        if canvas_width < 10 or canvas_height < 10:
            return

        # Calculer la taille de l'image zoomée
        orig_width, orig_height = self._graph_image.size
        new_width = int(orig_width * self._zoom_level)
        new_height = int(orig_height * self._zoom_level)

        if new_width > 0 and new_height > 0:
            resized = self._graph_image.resize((new_width, new_height), Image.LANCZOS)
            self._photo_image = ImageTk.PhotoImage(resized)

            # Position centrée + offset pan
            x = (canvas_width // 2) + self._pan_offset_x
            y = (canvas_height // 2) + self._pan_offset_y

            self.display_canvas.delete('all')
            self.display_canvas.create_image(x, y, image=self._photo_image, anchor='center')

    def _toggle_live(self):
        self.is_live = self.live_var.get()
        if self.is_live and self.on_update_callback:
            self.on_update_callback(self)

    def update_data(self, resultats, params, devise):
        """Met à jour les données si en mode Live"""
        if self.is_live:
            self.resultats = resultats
            self.params = params
            self.devise = devise
            self._draw_graph()

    def update_font_size(self, new_size):
        """Met à jour la taille de police et redessine"""
        self.font_size = new_size
        self._draw_graph()

    def _draw_graph(self):
        """Dessine le graphique et le rend en image"""
        self.ax.clear()

        if not self.resultats:
            self.ax.text(0.5, 0.5, "Pas de données", ha='center', va='center',
                        transform=self.ax.transAxes, fontsize=self.font_size)
        else:
            # Réutiliser la logique de GraphPanel
            panel = GraphPanel.__new__(GraphPanel)
            panel.ax = self.ax
            panel.current_graph_id = self.graph_id or 'enlarged'
            panel.current_graph_def = self.graph_def
            panel.resultats = self.resultats
            panel.params = self.params
            panel.devise = self.devise
            panel.font_size = self.font_size
            # Attributs nécessaires pour _draw_line_graph
            panel.curve_checkboxes = []
            panel.curve_visibility = {}
            panel.redraw = lambda: None
            # Créer un frame "dummy" pour curves_frame.pack_forget()
            panel.curves_frame = ttk.Frame(self)

            graph_type = self.graph_def.get('type', 'line')

            try:
                if graph_type == 'line' or graph_type == 'area':
                    panel._draw_line_graph(self.graph_def)
                elif graph_type == 'bar':
                    panel._draw_bar_graph(self.graph_def)
                elif graph_type == 'stacked_bar':
                    panel._draw_stacked_bar_graph(self.graph_def)
                else:
                    panel._draw_line_graph(self.graph_def)
            except Exception as e:
                self.ax.text(0.5, 0.5, f"Erreur: {e}",
                            ha='center', va='center', transform=self.ax.transAxes,
                            color='red', fontsize=self.font_size)

            titre = get_yaml_value(self.graph_def, 'title', 'titre', '').replace('{devise}', self.devise)
            self.ax.set_title(titre, fontsize=self.font_size + 2)
            self.ax.tick_params(axis='both', labelsize=self.font_size - 2)
            self.ax.grid(True, alpha=0.3)

            # Ajuster l'axe X pour réduire la marge gauche automatique de matplotlib
            xlim = self.ax.get_xlim()
            if graph_type in ('bar', 'stacked_bar'):
                # Pour les histogrammes: garder une petite marge pour la première barre
                self.ax.set_xlim(left=-0.4)
            elif xlim[0] < 5 and xlim[0] > -5:
                # Pour les graphiques en ligne commençant proche de 0
                self.ax.set_xlim(left=0)

        # Sauvegarder les limites d'axes originales pour le zoom/pan
        self._original_xlim = self.ax.get_xlim()
        self._original_ylim = self.ax.get_ylim()

        # Rendre en image
        self._render_to_image(reset_view=True)

    def _render_to_image(self, reset_view=True):
        """Rend le graphique matplotlib dans une image PIL"""
        canvas_width = self.display_canvas.winfo_width()
        canvas_height = self.display_canvas.winfo_height()

        if canvas_width < 50 or canvas_height < 50:
            canvas_width = 1000
            canvas_height = 700

        dpi = 100
        fig_width = canvas_width / dpi
        fig_height = canvas_height / dpi
        self.fig.set_size_inches(fig_width, fig_height)
        self.fig.tight_layout(pad=0.5)

        # Rendre en image PNG (sans bbox_inches='tight' pour préserver l'alignement)
        buf = io.BytesIO()
        self.fig.savefig(buf, format='png', dpi=dpi, facecolor='white',
                         edgecolor='none')
        buf.seek(0)

        self._graph_image = Image.open(buf).copy()
        buf.close()

        # Reset zoom/pan seulement si demandé
        if reset_view:
            self._zoom_level = 1.0
            self._pan_offset_x = 0
            self._pan_offset_y = 0

        self._update_display()

    def _generate_save_filename(self):
        """Génère un nom de fichier intelligent: scenario_graphtype_CODE=val_CODE=val"""
        parts = []

        scenario_name = ""
        if self.get_scenario_name_callback:
            scenario_name = self.get_scenario_name_callback()
        if scenario_name:
            scenario_name = Path(scenario_name).stem.replace(' ', '_')
            parts.append(scenario_name)

        if self.graph_id:
            parts.append(self.graph_id)

        if self.get_modified_params_callback:
            modified = self.get_modified_params_callback()
            for param_name, value in modified.items():
                code = PARAM_NAMES_TO_CODES.get(param_name)
                if code:
                    if isinstance(value, float) and value == int(value):
                        val_str = str(int(value))
                    elif isinstance(value, float):
                        val_str = f"{value:.1f}".rstrip('0').rstrip('.')
                    else:
                        val_str = str(value)
                    parts.append(f"{code}={val_str}")

        if not parts:
            return "graphique_enlarged"
        return "_".join(parts)

    def _save_graph(self):
        """Sauvegarde le graphique agrandi en SVG ou PNG"""
        if not self.resultats:
            messagebox.showwarning(t('dialogs.warning'), t('dialogs.no_graph_to_save'))
            return

        default_name = self._generate_save_filename()

        file_path = filedialog.asksaveasfilename(
            title=t('dialogs.save_graph_title'),
            initialfile=default_name,
            defaultextension=".svg",
            filetypes=[
                (t('file_types.svg'), "*.svg"),
                (t('file_types.png'), "*.png"),
                (t('file_types.pdf'), "*.pdf"),
                (t('file_types.jpeg'), "*.jpg"),
                (t('file_types.all'), "*.*")
            ]
        )

        if not file_path:
            return

        try:
            self.fig.savefig(file_path, dpi=150, bbox_inches='tight',
                           facecolor='white', edgecolor='none')
            self._set_status(t('status.graph_saved', path=file_path))
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_save', error=e))

    def _copy_graph_png(self):
        """Copie le graphique dans le presse-papier en PNG"""
        if not self.resultats:
            self._set_status(t('dialogs.no_graph_to_copy'), error=True)
            return

        try:
            buf = io.BytesIO()
            self.fig.savefig(buf, format='png', dpi=150, bbox_inches='tight',
                           facecolor='white', edgecolor='none')
            buf.seek(0)
            png_data = buf.getvalue()

            system = platform.system()
            if system == 'Linux':
                # Vérifier wl-copy (Wayland) ou xclip (X11)
                wl_copy_path = shutil.which('wl-copy')
                xclip_path = shutil.which('xclip')

                if wl_copy_path:
                    process = subprocess.Popen(
                        ['wl-copy', '--type', 'image/png'],
                        stdin=subprocess.PIPE,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL
                    )
                    process.communicate(png_data, timeout=5)
                elif xclip_path:
                    import tempfile
                    with tempfile.NamedTemporaryFile(suffix='.png', delete=False) as tmp:
                        tmp.write(png_data)
                        tmp_path = tmp.name

                    # Redirection stdin < fichier (méthode canonique pour images binaires)
                    subprocess.Popen(
                        f'xclip -selection clipboard -t image/png < "{tmp_path}" && sleep 60 && rm -f "{tmp_path}"',
                        shell=True,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL,
                        start_new_session=True
                    )
                else:
                    self._set_status("wl-copy ou xclip requis", error=True)
                    return

            elif system == 'Darwin':
                import tempfile
                with tempfile.NamedTemporaryFile(suffix='.png', delete=False) as tmp:
                    tmp.write(png_data)
                    tmp_path = tmp.name
                subprocess.run(['osascript', '-e',
                              f'set the clipboard to (read (POSIX file "{tmp_path}") as «class PNGf»)'],
                              check=True, timeout=5)
                os.unlink(tmp_path)
            elif system == 'Windows':
                try:
                    import win32clipboard
                    from PIL import Image
                    img = Image.open(io.BytesIO(png_data))
                    output = io.BytesIO()
                    img.convert('RGB').save(output, 'BMP')
                    data = output.getvalue()[14:]
                    output.close()
                    win32clipboard.OpenClipboard()
                    win32clipboard.EmptyClipboard()
                    win32clipboard.SetClipboardData(win32clipboard.CF_DIB, data)
                    win32clipboard.CloseClipboard()
                except ImportError:
                    self._set_status("pywin32 requis (pip install pywin32)", error=True)
                    return

            self._set_status(t('status.graph_copied_png'))
        except subprocess.TimeoutExpired:
            self._set_status("Timeout copie PNG", error=True)
        except Exception as e:
            self._set_status(f"Erreur copie: {e}", error=True)

    def _copy_graph_svg(self):
        """Copie le graphique dans le presse-papier en SVG (texte)"""
        if not self.resultats:
            self._set_status(t('dialogs.no_graph_to_copy'), error=True)
            return

        try:
            buf = io.BytesIO()
            self.fig.savefig(buf, format='svg', bbox_inches='tight',
                           facecolor='white', edgecolor='none')
            buf.seek(0)
            svg_content = buf.getvalue().decode('utf-8')

            system = platform.system()
            if system == 'Linux':
                wl_copy_path = shutil.which('wl-copy')
                xclip_path = shutil.which('xclip')

                if wl_copy_path:
                    process = subprocess.Popen(
                        ['wl-copy'],
                        stdin=subprocess.PIPE,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL
                    )
                    process.communicate(svg_content.encode('utf-8'), timeout=5)
                elif xclip_path:
                    import tempfile
                    with tempfile.NamedTemporaryFile(mode='w', suffix='.svg', delete=False, encoding='utf-8') as tmp:
                        tmp.write(svg_content)
                        tmp_path = tmp.name

                    subprocess.Popen(
                        f'xclip -selection clipboard -i "{tmp_path}" && sleep 60 && rm -f "{tmp_path}"',
                        shell=True,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL,
                        start_new_session=True
                    )
                else:
                    self.clipboard_clear()
                    self.clipboard_append(svg_content)
                    self.update()
            elif system == 'Darwin':
                process = subprocess.Popen(['pbcopy'], stdin=subprocess.PIPE)
                process.communicate(svg_content.encode('utf-8'), timeout=5)
            elif system == 'Windows':
                self.clipboard_clear()
                self.clipboard_append(svg_content)
                self.update()
            else:
                self.clipboard_clear()
                self.clipboard_append(svg_content)
                self.update()

            self._set_status(t('status.graph_copied_svg'))
        except subprocess.TimeoutExpired:
            self._set_status("Timeout copie SVG", error=True)
        except Exception as e:
            self._set_status(f"Erreur copie: {e}", error=True)


class SimulateurApp(tk.Tk):
    """Application principale"""

    def __init__(self):
        super().__init__()

        self.title(f"{t('app.title')} {t('app.version')}")
        self.geometry("1600x950")

        self.params = None
        self.base_params = None
        self.resultats = None
        self.devise = "€"
        self.graph_files = {}
        self.enlarged_windows = []  # Pour les fenêtres Live
        self.font_size = BASE_FONT_SIZE  # Taille de police ajustable
        self.grid_rows = 2  # Nombre de lignes de graphiques (1-4)
        self.grid_cols = 2  # Nombre de colonnes de graphiques (1-4)

        # Mode de l'application (normal ou comparison)
        self.current_mode = 'normal'  # 'normal' ou 'comparison'
        self.comparison_panel = None  # Panel du mode comparaison

        # Sauvegarder le scaling système par défaut (pour calculs ultérieurs)
        self.base_tk_scaling = self.tk.call('tk', 'scaling')

        # Géométrie de base (fixe, ne change pas avec la taille de police)
        self.base_width = 1600
        self.base_height = 950

        # Configurer les polices par défaut
        self._setup_default_fonts()

        self._load_graph_files()
        self._create_menu()
        self._create_widgets()
        self._load_scenarios_list()

        # Charger le premier scénario par défaut et lancer la simulation
        self.after(100, self._auto_start)

    def _setup_default_fonts(self):
        """Configure les polices par défaut et le scaling global (tkinter, menus et matplotlib)"""
        self._apply_scaling()

    def _apply_scaling(self):
        """Applique le scaling global basé sur la taille de police"""
        import tkinter.font as tkfont

        # Calculer le facteur de scaling basé sur la taille de police
        # Base: 12pt = scaling 1.0, donc 20pt = scaling ~1.67
        font_factor = self.font_size / 12.0  # 12pt est la taille de référence
        new_scaling = self.base_tk_scaling * font_factor

        # Appliquer le scaling global tk (affecte dialogues, widgets, etc.)
        self.tk.call('tk', 'scaling', new_scaling)

        # Polices tkinter
        default_font = tkfont.nametofont("TkDefaultFont")
        default_font.configure(size=self.font_size)
        text_font = tkfont.nametofont("TkTextFont")
        text_font.configure(size=self.font_size)
        fixed_font = tkfont.nametofont("TkFixedFont")
        fixed_font.configure(size=self.font_size)
        menu_font = tkfont.nametofont("TkMenuFont")
        menu_font.configure(size=self.font_size)

        # Polices matplotlib pour les graphiques
        plt.rcParams.update({
            'font.size': self.font_size,
            'axes.titlesize': self.font_size + 2,
            'axes.labelsize': self.font_size,
            'xtick.labelsize': self.font_size - 2,
            'ytick.labelsize': self.font_size - 2,
            'legend.fontsize': self.font_size - 2,
        })

        # Configurer les styles ttk pour un meilleur scaling
        style = ttk.Style()
        style.configure('.', font=('TkDefaultFont', self.font_size))
        style.configure('TButton', padding=(self.font_size // 2, self.font_size // 4))
        style.configure('TCombobox', padding=(self.font_size // 4,))
        style.configure('TEntry', padding=(self.font_size // 4,))

    def _update_geometry(self):
        """Met à jour la géométrie de la fenêtre selon la taille de police"""
        # Facteur basé sur 12pt comme référence
        factor = self.font_size / 12.0
        new_width = int(self.base_width * factor)
        new_height = int(self.base_height * factor)

        # Limiter à la taille de l'écran
        screen_width = self.winfo_screenwidth()
        screen_height = self.winfo_screenheight()
        new_width = min(new_width, screen_width - 50)
        new_height = min(new_height, screen_height - 100)

        self.geometry(f"{new_width}x{new_height}")

    def _update_title(self):
        """Met à jour le titre de la fenêtre avec le nom du scénario"""
        base_title = f"{t('app.title')} {t('app.version')}"

        # Ajouter le nom du scénario si disponible
        if self.base_params:
            scenario_name = self.base_params.nom_pays
            base_title = f"{scenario_name} ({self.devise}) - {base_title}"

        # Ajouter le facteur de police si différent de 12pt
        if self.font_size != 12:
            factor = self.font_size / 12.0
            base_title = f"{base_title} [{self.font_size}pt x{factor:.1f}]"

        self.title(base_title)

    def _auto_start(self):
        """Charge le premier scénario et lance la simulation automatiquement"""
        if self.scenarios_list:
            self._select_scenario(self.scenarios_list[0])

    def _add_tooltip(self, widget, text):
        """Ajoute un tooltip à un widget"""
        def show_tooltip(event):
            tooltip = tk.Toplevel(widget)
            tooltip.wm_overrideredirect(True)
            tooltip.wm_geometry(f"+{event.x_root+10}+{event.y_root+10}")
            label = tk.Label(tooltip, text=text, background="#ffffe0", relief='solid', borderwidth=1,
                           font=('TkDefaultFont', self.font_size - 2))
            label.pack()
            widget._tooltip = tooltip
            widget.after(2000, lambda: tooltip.destroy() if hasattr(widget, '_tooltip') else None)

        def hide_tooltip(event):
            if hasattr(widget, '_tooltip') and widget._tooltip:
                widget._tooltip.destroy()
                widget._tooltip = None

        widget.bind('<Enter>', show_tooltip)
        widget.bind('<Leave>', hide_tooltip)

    def _load_graph_files(self):
        self.graph_files = load_graph_files()

    def _create_menu(self):
        self.menubar = tk.Menu(self)

        self.file_menu = tk.Menu(self.menubar, tearoff=0)
        self.file_menu.add_command(label=t('menu.file_load_scenario'), command=self._load_scenario_dialog)
        self.file_menu.add_separator()
        self.file_menu.add_command(label=t('menu.file_quit'), command=self.quit)
        self.menubar.add_cascade(label=t('menu.file'), menu=self.file_menu)

        self.graph_menu = tk.Menu(self.menubar, tearoff=0)
        self.graph_menu.add_command(label=t('menu.graphs_new'), command=self._create_new_graph)
        self.graph_menu.add_command(label=t('menu.graphs_reload'), command=self._reload_graphs)
        self.graph_menu.add_separator()
        self.graph_menu.add_command(label=t('menu.graphs_open_folder'), command=self._open_graphs_folder)
        self.menubar.add_cascade(label=t('menu.graphs'), menu=self.graph_menu)

        self.scenario_menu = tk.Menu(self.menubar, tearoff=0)
        # Le menu sera rempli dynamiquement par _update_scenario_menu
        self.menubar.add_cascade(label=t('menu.scenarios'), menu=self.scenario_menu)

        # Menu Affichage - tout en ligne (pas de sous-menus)
        self.view_menu = tk.Menu(self.menubar, tearoff=0)
        self._update_view_menu()
        self.menubar.add_cascade(label=t('menu.view'), menu=self.view_menu)

        # Menu Mode (Normal / Comparaison)
        if COMPARISON_MODE_AVAILABLE:
            self.mode_menu = tk.Menu(self.menubar, tearoff=0)
            self.mode_menu.add_command(
                label="📊 Mode Normal",
                command=lambda: self._switch_mode('normal')
            )
            self.mode_menu.add_command(
                label="⚖️ Mode Comparaison",
                command=lambda: self._switch_mode('comparison')
            )
            self.menubar.add_cascade(label="Mode", menu=self.mode_menu)
        else:
            self.mode_menu = None

        # Menu Langue
        lang_menu = tk.Menu(self.menubar, tearoff=0)
        for lang_code, lang_name in get_available_languages():
            lang_menu.add_command(
                label=lang_name,
                command=lambda lc=lang_code: self._change_language(lc)
            )
        self.menubar.add_cascade(label=t('menu.language'), menu=lang_menu)

        help_menu = tk.Menu(self.menubar, tearoff=0)
        help_menu.add_command(label=t('menu.help_about'), command=self._show_about)
        self.menubar.add_cascade(label=t('menu.help'), menu=help_menu)

        self.config(menu=self.menubar)

        # Raccourcis clavier pour la taille de police
        self.bind('<Control-plus>', lambda e: self._change_font_size(self.font_size + 1))
        self.bind('<Control-minus>', lambda e: self._change_font_size(self.font_size - 1))
        self.bind('<Control-equal>', lambda e: self._change_font_size(self.font_size + 1))  # Pour + sans Shift
        self.bind('<Control-KP_Add>', lambda e: self._change_font_size(self.font_size + 1))  # Pavé numérique
        self.bind('<Control-KP_Subtract>', lambda e: self._change_font_size(self.font_size - 1))

    def _create_widgets(self):
        # Barre de statut en bas (créée en premier pour être en bas)
        self.status_frame = ttk.Frame(self)
        self.status_frame.pack(side='bottom', fill='x')
        self.status_var = tk.StringVar(value=t('status.ready'))
        # Utiliser tk.Label au lieu de ttk.Label pour supporter les couleurs
        self.status_label = tk.Label(self.status_frame, textvariable=self.status_var,
                                     relief='sunken', anchor='w', padx=5, pady=2,
                                     bg='#f0f0f0', fg='black')
        self.status_label.pack(fill='x')
        self._status_clear_job = None  # Pour effacer le statut après un délai

        self.main_paned = ttk.PanedWindow(self, orient='horizontal')
        self.main_paned.pack(fill='both', expand=True, padx=5, pady=5)

        # Panel gauche
        self.left_frame = ttk.Frame(self.main_paned, width=380)
        self.main_paned.add(self.left_frame, weight=0)

        # Variable pour le scénario actuel (sélection via menu)
        self.scenario_var = tk.StringVar()
        self.scenarios_list = []  # Liste des scénarios disponibles

        # === PARAMÈTRES (avec bouton crayon pour éditer scénario) ===
        self.param_panel = ParameterPanel(
            self.left_frame,
            self._on_param_change,
            font_size=self.font_size,
            on_edit_scenario=self._edit_current_scenario,
            on_scenario_reload=self._reload_current_scenario
        )
        self.param_panel.pack(fill='x', padx=3, pady=3)

        # === AIDE (scrollable) ===
        self.help_frame = ttk.LabelFrame(self.left_frame, text=f"📖 {t('panel.help')}")
        self.help_frame.pack(fill='both', expand=True, padx=3, pady=3)

        self.help_text = tk.Text(self.help_frame, wrap='word',
                                 font=('Consolas', self.font_size - 1),
                                 width=40)
        help_scroll = ttk.Scrollbar(self.help_frame, orient='vertical', command=self.help_text.yview)
        self.help_text.configure(yscrollcommand=help_scroll.set)
        help_scroll.pack(side='right', fill='y')
        self.help_text.pack(side='left', fill='both', expand=True)
        self.help_text.insert('1.0', HELP_TEXT)
        self.help_text.configure(state='disabled')

        # Binding Select All (Ctrl+A) pour le texte d'aide
        self.help_text.bind('<Control-a>', self._select_all_help_text)
        self.help_text.bind('<Control-A>', self._select_all_help_text)

        # Panel droit : graphiques
        self.right_frame = ttk.Frame(self.main_paned)
        self.main_paned.add(self.right_frame, weight=1)

        # Frame pour les 4 petits graphiques (grille 2x2)
        self.graphs_frame = ttk.Frame(self.right_frame)
        self.graphs_frame.pack(fill='both', expand=True)

        self.graphs_frame.columnconfigure(0, weight=1)
        self.graphs_frame.columnconfigure(1, weight=1)
        self.graphs_frame.rowconfigure(0, weight=1)
        self.graphs_frame.rowconfigure(1, weight=1)

        # Frame pour le graphique agrandi in-situ (caché par défaut)
        self.expanded_frame = ttk.Frame(self.right_frame)
        self.expanded_graph_panel = None
        self.expanded_source_index = None

        self.graph_panels = []
        self.default_graphs = ['evolution_pib', 'evolution_dettes_pct', 'effet_combine_an0', 'effet_combine_final',
                               'evolution_pouvoir_achat', 'evolution_interets', 'evolution_differentiel', 'evolution_retraites',
                               'gain_salaires_final', 'dette_publique_nominale', 'dette_implicite', 'flux_pensions',
                               'evolution_pouvoir_achat_transition', 'comparaison_prelevements', 'repartition_depenses', 'taux_prelevement']
        self._rebuild_graph_grid()

        # Frame pour le mode comparaison (caché par défaut)
        self.comparison_frame = ttk.Frame(self.right_frame)
        # Ne pas pack - sera affiché lors du basculement de mode

    def _rebuild_graph_grid(self):
        """Reconstruit la grille de panneaux graphiques selon grid_rows et grid_cols"""
        # Sauvegarder les graphiques par position (row, col) pour conserver leur emplacement
        current_graphs = {}
        for panel in self.graph_panels:
            if hasattr(panel, 'grid_info'):
                info = panel.grid_info()
                if info:
                    row = info.get('row', 0)
                    col = info.get('column', 0)
                    graph_id = panel.current_graph_id if hasattr(panel, 'current_graph_id') else None
                    current_graphs[(row, col)] = graph_id

        # Supprimer tous les widgets de la grille
        for widget in self.graphs_frame.winfo_children():
            widget.destroy()
        self.graph_panels = []

        # Reconfigurer la grille
        for i in range(6):  # Max 4 + 1 pour les boutons
            self.graphs_frame.columnconfigure(i, weight=0)
            self.graphs_frame.rowconfigure(i, weight=0)
        for col in range(self.grid_cols):
            self.graphs_frame.columnconfigure(col, weight=1)
        for row in range(self.grid_rows):
            self.graphs_frame.rowconfigure(row, weight=1)

        # Créer les panneaux graphiques
        panel_index = 0
        for row in range(self.grid_rows):
            for col in range(self.grid_cols):
                panel = GraphPanel(self.graphs_frame, self._on_enlarge_graph, self._on_edit_graph, panel_index,
                                  font_size=self.font_size, on_graph_reload=self._reload_graphs,
                                  get_modified_params_callback=self._get_modified_params,
                                  get_scenario_name_callback=self._get_scenario_name,
                                  on_new_graph=self._create_new_graph,
                                  on_expand_inplace=self._on_expand_inplace,
                                  on_status_update=self._set_status)
                panel.grid(row=row, column=col, sticky='nsew', padx=2, pady=2)
                panel.update_graph_files(self.graph_files)
                self.graph_panels.append(panel)

                # Restaurer le graphique à sa position exacte, ou utiliser un défaut
                if (row, col) in current_graphs and current_graphs[(row, col)]:
                    panel.set_graph(current_graphs[(row, col)])
                elif panel_index < len(self.default_graphs) and self.default_graphs[panel_index] in self.graph_files:
                    panel.set_graph(self.default_graphs[panel_index])

                panel_index += 1

        # Colonne de boutons à droite (verticale) pour gérer les colonnes
        btn_col = self.grid_cols
        btn_frame_col = ttk.Frame(self.graphs_frame)
        btn_frame_col.grid(row=0, column=btn_col, rowspan=self.grid_rows, sticky='ns', padx=2, pady=2)

        if self.grid_cols < 4:
            add_col_btn = ttk.Button(btn_frame_col, text="[+]", width=3, command=self._add_column)
            add_col_btn.pack(side='top', pady=2)
            self._add_tooltip(add_col_btn, "Ajouter une colonne")

        if self.grid_cols > 1:
            remove_col_btn = ttk.Button(btn_frame_col, text="[−]", width=3, command=self._remove_column)
            remove_col_btn.pack(side='bottom', pady=2)
            self._add_tooltip(remove_col_btn, "Supprimer la dernière colonne")

        # Ligne de boutons en bas (horizontale) pour gérer les lignes
        btn_row = self.grid_rows
        btn_frame_row = ttk.Frame(self.graphs_frame)
        btn_frame_row.grid(row=btn_row, column=0, columnspan=self.grid_cols, sticky='ew', padx=2, pady=2)

        if self.grid_rows < 4:
            add_row_btn = ttk.Button(btn_frame_row, text="[+]", width=3, command=self._add_row)
            add_row_btn.pack(side='left', padx=2)
            self._add_tooltip(add_row_btn, "Ajouter une ligne")

        if self.grid_rows > 1:
            remove_row_btn = ttk.Button(btn_frame_row, text="[−]", width=3, command=self._remove_row)
            remove_row_btn.pack(side='right', padx=2)
            self._add_tooltip(remove_row_btn, "Supprimer la dernière ligne")

        # Mettre à jour avec les résultats actuels si disponibles
        if self.resultats:
            for panel in self.graph_panels:
                panel.set_data(self.resultats, self.params, self.devise)

    def _add_column(self):
        """Ajoute une colonne de graphiques à droite"""
        if self.grid_cols < 4:
            self.grid_cols += 1
            self._rebuild_graph_grid()

    def _remove_column(self):
        """Supprime la dernière colonne de graphiques (à droite)"""
        if self.grid_cols > 1:
            self.grid_cols -= 1
            self._rebuild_graph_grid()

    def _add_row(self):
        """Ajoute une ligne de graphiques en bas"""
        if self.grid_rows < 4:
            self.grid_rows += 1
            self._rebuild_graph_grid()

    def _remove_row(self):
        """Supprime la dernière ligne de graphiques (en bas)"""
        if self.grid_rows > 1:
            self.grid_rows -= 1
            self._rebuild_graph_grid()

    def _change_grid_rows(self, rows):
        """Change le nombre de lignes de la grille"""
        if rows != self.grid_rows:
            self.grid_rows = rows
            self._rebuild_graph_grid()
            self._update_view_menu()

    def _change_grid_cols(self, cols):
        """Change le nombre de colonnes de la grille"""
        if cols != self.grid_cols:
            self.grid_cols = cols
            self._rebuild_graph_grid()
            self._update_view_menu()

    def _switch_mode(self, mode, scenario_path=None):
        """Bascule entre le mode normal et le mode comparaison"""
        if mode == self.current_mode and scenario_path is None:
            return

        if mode == 'comparison':
            # Passer en mode comparaison
            self.graphs_frame.pack_forget()
            self.expanded_frame.pack_forget()

            # Masquer le panneau gauche (paramètres et aide)
            try:
                self.main_paned.forget(self.left_frame)
            except:
                pass

            # Créer le panneau de comparaison si nécessaire
            if self.comparison_panel is None and COMPARISON_MODE_AVAILABLE:
                # Préparer la configuration initiale
                available = get_available_scenarios(CONFIG_DIR)
                if len(available) < 2:
                    messagebox.showwarning(
                        "Mode Comparaison",
                        "Il faut au moins 2 scénarios pour utiliser le mode comparaison."
                    )
                    # Réafficher le panneau gauche
                    self.main_paned.insert(0, self.left_frame, weight=0)
                    return

                # Sélectionner les 2-3 premiers scénarios par défaut
                initial_scenarios = [s[0] for s in available[:min(3, len(available))]]

                # Utiliser les mêmes graphiques YAML que le mode normal
                initial_graphs = self.default_graphs[:2] if len(self.default_graphs) >= 2 else self.default_graphs[:1]

                config = ComparisonConfig(
                    scenarios=initial_scenarios,
                    graphs=initial_graphs  # Mêmes graphiques YAML qu'en mode normal
                )

                self.comparison_panel = ComparisonGridPanel(
                    self.comparison_frame,
                    config=config,
                    available_scenarios=available,
                    graph_files=self.graph_files,
                    load_scenario_callback=self._load_scenario_for_comparison,
                    GraphPanelClass=GraphPanel,
                    font_size=self.font_size,
                    on_status_update=self._set_status,
                    add_tooltip_callback=self._add_tooltip,
                    on_switch_to_normal=self._switch_to_normal_with_scenario
                )
                self.comparison_panel.pack(fill='both', expand=True)

            self.comparison_frame.pack(fill='both', expand=True)
            self.current_mode = 'comparison'
            self._set_status("Mode Comparaison activé")

        else:  # mode == 'normal'
            # Revenir au mode normal
            self.comparison_frame.pack_forget()

            # Réafficher le panneau gauche
            try:
                self.main_paned.insert(0, self.left_frame, weight=0)
            except:
                pass

            self.graphs_frame.pack(fill='both', expand=True)
            self.current_mode = 'normal'

            # Charger le scénario spécifié si fourni
            if scenario_path:
                self._load_scenario(scenario_path)
                self._set_status(f"Mode Normal - {Path(scenario_path).stem}")
            else:
                # Rafraîchir les graphiques avec les données actuelles
                if self.resultats is not None:
                    self._update_graphs()
                self._set_status("Mode Normal activé")

        self._update_menus_for_mode()
        self._update_title()

    def _switch_to_normal_with_scenario(self, scenario_path):
        """Callback pour passer en mode normal avec un scénario spécifique"""
        self._switch_mode('normal', scenario_path=scenario_path)

    def _update_menus_for_mode(self):
        """Active/désactive les menus selon le mode actuel"""
        is_comparison = (self.current_mode == 'comparison')

        # En mode comparaison, désactiver les menus qui n'ont pas de sens:
        # - Scénarios: les scénarios sont sélectionnés par colonne
        # - Graphiques (partiellement): les graphiques sont sélectionnés par ligne

        # Désactiver le menu Scénarios en mode comparaison
        # On utilise entryconfigure sur la menubar pour changer l'état
        try:
            # Trouver l'index du menu Scénarios
            scenario_idx = None
            graph_idx = None
            for i in range(self.menubar.index('end') + 1):
                try:
                    label = self.menubar.entrycget(i, 'label')
                    if 'cénario' in label:  # Scénarios / Scenarios
                        scenario_idx = i
                    elif 'raph' in label:  # Graphiques / Graphs
                        graph_idx = i
                except:
                    pass

            # Désactiver/activer le menu Scénarios
            if scenario_idx is not None:
                state = 'disabled' if is_comparison else 'normal'
                self.menubar.entryconfigure(scenario_idx, state=state)

            # Désactiver/activer certaines entrées du menu Graphiques
            # Garder "Ouvrir dossier" actif, mais désactiver "Nouveau" et "Recharger"
            if graph_idx is not None and is_comparison:
                # En mode comparaison, désactiver les 2 premières entrées
                try:
                    self.graph_menu.entryconfigure(0, state='disabled')  # Nouveau
                    self.graph_menu.entryconfigure(1, state='disabled')  # Recharger
                except:
                    pass
            elif graph_idx is not None:
                # En mode normal, tout activer
                try:
                    self.graph_menu.entryconfigure(0, state='normal')
                    self.graph_menu.entryconfigure(1, state='normal')
                except:
                    pass

        except Exception as e:
            # Silencieux en cas d'erreur
            pass

    def _load_scenario_for_comparison(self, ini_path):
        """
        Charge un scénario pour le mode comparaison.
        Retourne (params, resultats, devise)
        """
        try:
            params = charger_configuration(ini_path)
            sim = SimulateurTransition(params)
            resultats = sim.simuler()
            resultats = calculer_toutes_metriques_indexees(resultats)

            # Déterminer la devise
            devise = "€"
            if hasattr(params, 'devise'):
                devise = params.devise

            return params, resultats, devise

        except Exception as e:
            self._set_status(f"Erreur chargement {ini_path}: {e}", error=True)
            raise

    def _update_view_menu(self):
        """Met à jour le menu Affichage avec les options de police"""
        self.view_menu.delete(0, 'end')

        # Section police
        self.view_menu.add_command(label=t('menu.view_font'), state='disabled')

        # Tailles de police
        for size in [10, 12, 14, 16, 18, 20]:
            marker = "● " if size == self.font_size else "○ "
            self.view_menu.add_command(
                label=f"  {marker}{size} pt",
                command=lambda s=size: self._change_font_size(s)
            )

        self.view_menu.add_separator()

        # Raccourcis police
        self.view_menu.add_command(
            label=f"  {t('menu.view_font_larger')}",
            command=lambda: self._change_font_size(self.font_size + 1)
        )
        self.view_menu.add_command(
            label=f"  {t('menu.view_font_smaller')}",
            command=lambda: self._change_font_size(self.font_size - 1)
        )
        self.view_menu.add_command(
            label=f"  {t('menu.view_font_default')}",
            command=lambda: self._change_font_size(BASE_FONT_SIZE)
        )

    def _load_scenarios_list(self):
        """Charge la liste des scénarios et met à jour le menu"""
        self.scenarios_list = []
        if CONFIG_DIR.exists():
            for item in CONFIG_DIR.rglob('*.ini'):
                rel_path = item.relative_to(CONFIG_DIR)
                self.scenarios_list.append(str(rel_path))
        self.scenarios_list.sort()
        self._update_scenario_menu()

    def _set_status(self, message, timeout=5000, error=False):
        """Met à jour la barre de statut avec un message temporaire

        Args:
            message: Le message à afficher
            timeout: Délai en ms avant effacement (défaut 5000)
            error: Si True, affiche en rouge
        """
        # Annuler le job précédent s'il existe
        if self._status_clear_job:
            self.after_cancel(self._status_clear_job)
        # Afficher le message avec la couleur appropriée
        self.status_var.set(message)
        self.status_label.configure(fg='#dc2626' if error else 'black')
        # Programmer l'effacement après le délai
        def clear_status():
            self.status_var.set(t('status.ready'))
            self.status_label.configure(fg='black')
        self._status_clear_job = self.after(timeout, clear_status)

    def _update_scenario_menu(self):
        """Met à jour le menu Scénario avec le scénario actuel et la liste"""
        self.scenario_menu.delete(0, 'end')

        # 1. Scénario actuel (en gras, désactivé)
        current = self.scenario_var.get()
        if current:
            display_name = Path(current).stem
            self.scenario_menu.add_command(
                label=f"▶ {display_name}",
                state='disabled'
            )
        else:
            self.scenario_menu.add_command(
                label=t('menu.scenarios_none_selected'),
                state='disabled'
            )

        # 2. Séparateur
        self.scenario_menu.add_separator()

        # 3. Éditer le scénario actuel
        self.scenario_menu.add_command(
            label=t('menu.scenarios_edit_current'),
            command=self._edit_current_scenario
        )

        # 4. Séparateur
        self.scenario_menu.add_separator()

        # 5. Liste des scénarios disponibles
        for scenario in self.scenarios_list:
            # Afficher juste le nom sans extension
            display_name = Path(scenario).stem
            # Marquer le scénario actuel avec une coche
            label = f"✓ {display_name}" if scenario == current else f"   {display_name}"
            self.scenario_menu.add_command(
                label=label,
                command=lambda s=scenario: self._select_scenario(s)
            )

    def _select_scenario(self, scenario_rel):
        """Sélectionne un scénario depuis le menu"""
        self.scenario_var.set(scenario_rel)
        scenario_path = CONFIG_DIR / scenario_rel
        self._load_scenario(scenario_path)
        # Mettre à jour le menu pour refléter la sélection
        self._update_scenario_menu()

    def _on_scenario_selected(self, event=None):
        """Appelé quand un scénario est sélectionné (compatibilité)"""
        scenario_rel = self.scenario_var.get()
        if scenario_rel:
            scenario_path = CONFIG_DIR / scenario_rel
            self._load_scenario(scenario_path)

    def _load_scenario(self, path):
        try:
            self.base_params = charger_configuration(str(path))
            self.params = self.base_params
            self.param_panel.load_params(self.base_params)

            # Mémoriser le chemin du scénario pour l'édition
            self.param_panel.set_scenario_file(path)

            config = configparser.ConfigParser()
            config.read(str(path), encoding='utf-8')
            self.devise = config.get('Pays', 'devise', fallback='€')

            # Mettre à jour la devise dans le panneau
            self.param_panel.set_devise(self.devise)

            # Mettre le nom du scénario dans le titre de la fenêtre
            self._update_title()

            # Lancer la simulation automatiquement
            self._run_simulation()
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_load', error=e))

    def _load_scenario_dialog(self):
        path = filedialog.askopenfilename(
            title=t('dialogs.load_scenario_title'),
            initialdir=CONFIG_DIR,
            filetypes=[(t('file_types.ini'), "*.ini"), (t('file_types.all'), "*.*")]
        )
        if path:
            self._load_scenario(path)

    def _select_all_help_text(self, event):
        """Sélectionne tout le texte dans le panneau d'aide"""
        self.help_text.tag_add('sel', '1.0', 'end')
        self.help_text.mark_set('insert', 'end')
        return 'break'

    def _on_param_change(self):
        """Appelé quand une valeur change - relance la simulation automatiquement"""
        if self.base_params:
            self.params = self.param_panel.get_modified_params(self.base_params)
            # Récupérer la devise du panneau
            self.devise = self.param_panel.get_devise()
            self._run_simulation()

    def _run_simulation(self):
        if not self.base_params:
            return

        try:
            params = self.param_panel.get_modified_params(self.base_params)
            sim = SimulateurTransition(params)
            self.resultats = sim.simuler()
            # Calculer les métriques indexées (base 100) pour les graphiques comparatifs
            self.resultats = calculer_toutes_metriques_indexees(self.resultats)
            self.params = params

            self._update_graphs()

            # Mettre à jour les fenêtres Live
            self._update_live_windows()

        except Exception as e:
            print(f"Simulation error: {e}")
            messagebox.showerror(t('dialogs.simulation_error'), str(e))

    def _update_graphs(self):
        for panel in self.graph_panels:
            panel.set_data(self.resultats, self.params, self.devise)

    def _update_live_windows(self):
        """Met à jour les fenêtres agrandies en mode Live"""
        for window in self.enlarged_windows[:]:
            try:
                if window.winfo_exists():
                    window.update_data(self.resultats, self.params, self.devise)
                else:
                    self.enlarged_windows.remove(window)
            except:
                self.enlarged_windows.remove(window)

    def _register_live_window(self, window):
        """Enregistre une fenêtre pour les mises à jour Live"""
        self.enlarged_windows.append(window)

    def _on_enlarge_graph(self, panel_index, graph_id, graph_def):
        if self.resultats and graph_def:
            window = EnlargedGraphWindow(
                self, graph_def, self.resultats, self.params, self.devise,
                font_size=self.font_size,
                on_update_callback=self._register_live_window,
                get_modified_params_callback=self._get_modified_params,
                get_scenario_name_callback=self._get_scenario_name,
                graph_id=graph_id
            )
            self.enlarged_windows.append(window)

    def _on_expand_inplace(self, panel_index, graph_id, graph_def):
        """Agrandit le graphique dans la fenêtre principale (in-situ) avec zoom/pan image"""
        if not self.resultats or not graph_def:
            return

        # Sauvegarder l'index source pour pouvoir revenir
        self.expanded_source_index = panel_index

        # Cacher la grille des 4 graphiques
        self.graphs_frame.pack_forget()

        # Créer le contenu de l'expanded frame
        for widget in self.expanded_frame.winfo_children():
            widget.destroy()

        # Barre d'outils avec bouton minimiser
        toolbar = ttk.Frame(self.expanded_frame)
        toolbar.pack(fill='x', padx=5, pady=5)

        titre = get_yaml_value(graph_def, 'title', 'titre', graph_id)
        title_label = ttk.Label(toolbar, text=titre, font=('TkDefaultFont', self.font_size + 2, 'bold'))
        title_label.pack(side='left', padx=10)

        # Instructions zoom/pan
        ttk.Label(toolbar, text="🔍 Molette=zoom | Drag=pan | Double-clic droit=reset",
                  font=('TkDefaultFont', 9)).pack(side='left', padx=20)

        # Boutons icônes avec tooltips (comme GraphPanel)
        minimize_btn = ttk.Button(toolbar, text="⬜", width=3, command=self._minimize_expanded)
        minimize_btn.pack(side='right', padx=2)
        self._add_tooltip(minimize_btn, t('tooltip.minimize'))

        save_btn = ttk.Button(toolbar, text="💾", width=3, command=lambda: self._save_expanded_graph(graph_id))
        save_btn.pack(side='right', padx=2)
        self._add_tooltip(save_btn, t('tooltip.save_graph'))

        copy_png_btn = ttk.Button(toolbar, text="📋", width=3, command=self._copy_expanded_graph_png)
        copy_png_btn.pack(side='right', padx=2)
        self._add_tooltip(copy_png_btn, t('tooltip.copy_png'))

        copy_svg_btn = ttk.Button(toolbar, text="📄", width=3, command=self._copy_expanded_graph_svg)
        copy_svg_btn.pack(side='right', padx=2)
        self._add_tooltip(copy_svg_btn, t('tooltip.copy_svg'))

        reset_zoom_btn = ttk.Button(toolbar, text="⟲", width=3, command=self._expanded_reset_zoom)
        reset_zoom_btn.pack(side='right', padx=2)
        self._add_tooltip(reset_zoom_btn, t('tooltip.reset_zoom'))

        # Créer le graphique avec système zoom/pan image
        from matplotlib.figure import Figure

        self.expanded_fig = Figure(figsize=(12, 8), dpi=100)
        self.expanded_ax = self.expanded_fig.add_subplot(111)

        # Canvas tkinter pour zoom/pan image (au lieu de FigureCanvasTkAgg)
        self.expanded_display_canvas = tk.Canvas(self.expanded_frame, bg='white', highlightthickness=0)
        self.expanded_display_canvas.pack(fill='both', expand=True, padx=10, pady=5)

        # État zoom/pan pour expanded
        self._expanded_graph_image = None
        self._expanded_photo_image = None
        self._expanded_zoom_level = 1.0
        self._expanded_pan_offset_x = 0
        self._expanded_pan_offset_y = 0
        self._expanded_drag_start_x = 0
        self._expanded_drag_start_y = 0

        # Bind pour zoom/pan
        self.expanded_display_canvas.bind('<MouseWheel>', self._on_expanded_mousewheel)
        self.expanded_display_canvas.bind('<Button-4>', lambda e: self._expanded_zoom(1.15))
        self.expanded_display_canvas.bind('<Button-5>', lambda e: self._expanded_zoom(0.87))
        self.expanded_display_canvas.bind('<ButtonPress-1>', self._on_expanded_drag_start)
        self.expanded_display_canvas.bind('<B1-Motion>', self._on_expanded_drag_motion)
        self.expanded_display_canvas.bind('<Double-Button-3>', lambda e: self._expanded_reset_zoom())
        self.expanded_display_canvas.bind('<Configure>', lambda e: self._draw_expanded_graph(graph_def, graph_id))

        # Afficher l'expanded frame
        self.expanded_frame.pack(fill='both', expand=True)
        self.expanded_graph_def = graph_def
        self.expanded_graph_id = graph_id

        # Bind Échap pour revenir aux 4 panneaux
        self.bind('<Escape>', lambda e: self._minimize_expanded())

        # Dessiner le graphique après un délai
        self.after(100, lambda: self._draw_expanded_graph(graph_def, graph_id))

    def _on_expanded_mousewheel(self, event):
        """Zoom avec molette pour expanded view"""
        if event.delta > 0:
            self._expanded_zoom(1.15)
        else:
            self._expanded_zoom(0.87)

    def _expanded_zoom(self, factor):
        """Zoom sur l'image expanded"""
        if not self._expanded_graph_image:
            return
        new_zoom = self._expanded_zoom_level * factor
        if new_zoom < 0.3 or new_zoom > 5.0:
            return
        self._expanded_zoom_level = new_zoom
        self._update_expanded_display()

    def _expanded_reset_zoom(self):
        """Reset zoom/pan expanded"""
        self._expanded_zoom_level = 1.0
        self._expanded_pan_offset_x = 0
        self._expanded_pan_offset_y = 0
        self._update_expanded_display()

    def _on_expanded_drag_start(self, event):
        """Début drag expanded"""
        self._expanded_drag_start_x = event.x
        self._expanded_drag_start_y = event.y

    def _on_expanded_drag_motion(self, event):
        """Drag motion expanded"""
        if not self._expanded_graph_image:
            return
        dx = event.x - self._expanded_drag_start_x
        dy = event.y - self._expanded_drag_start_y
        self._expanded_pan_offset_x += dx
        self._expanded_pan_offset_y += dy
        self._expanded_drag_start_x = event.x
        self._expanded_drag_start_y = event.y
        self._update_expanded_display()

    def _update_expanded_display(self):
        """Met à jour l'affichage expanded avec zoom/pan"""
        if not self._expanded_graph_image:
            return
        canvas_width = self.expanded_display_canvas.winfo_width()
        canvas_height = self.expanded_display_canvas.winfo_height()
        if canvas_width < 10 or canvas_height < 10:
            return
        orig_width, orig_height = self._expanded_graph_image.size
        new_width = int(orig_width * self._expanded_zoom_level)
        new_height = int(orig_height * self._expanded_zoom_level)
        if new_width > 0 and new_height > 0:
            resized = self._expanded_graph_image.resize((new_width, new_height), Image.LANCZOS)
            self._expanded_photo_image = ImageTk.PhotoImage(resized)
            x = (canvas_width // 2) + self._expanded_pan_offset_x
            y = (canvas_height // 2) + self._expanded_pan_offset_y
            self.expanded_display_canvas.delete('all')
            self.expanded_display_canvas.create_image(x, y, image=self._expanded_photo_image, anchor='center')

    def _draw_expanded_graph(self, graph_def, graph_id):
        """Dessine le graphique dans l'expanded view et rend en image"""
        self.expanded_ax.clear()

        graph_type = get_yaml_value(graph_def, 'type', 'type', 'line')

        if graph_type == 'line':
            self._draw_expanded_line_graph(graph_def)
        elif graph_type in ('bar', 'stacked_bar'):
            self._draw_expanded_bar_graph(graph_def, graph_id, stacked=(graph_type == 'stacked_bar'))
        elif graph_type == 'salary_evolution':
            self._draw_expanded_salary_evolution_graph(graph_def)

        titre = get_yaml_value(graph_def, 'title', 'titre', graph_id).replace('{devise}', self.devise)
        self.expanded_ax.set_title(titre, fontsize=self.font_size + 2)
        self.expanded_ax.grid(True, alpha=0.3)

        # Ajuster la taille de la figure au canvas
        canvas_width = self.expanded_display_canvas.winfo_width()
        canvas_height = self.expanded_display_canvas.winfo_height()
        if canvas_width < 50 or canvas_height < 50:
            canvas_width = 800
            canvas_height = 600

        dpi = 100
        self.expanded_fig.set_size_inches(canvas_width / dpi, canvas_height / dpi)
        self.expanded_fig.tight_layout()

        # Rendre en image PNG
        buf = io.BytesIO()
        self.expanded_fig.savefig(buf, format='png', dpi=dpi, facecolor='white',
                                  edgecolor='none', bbox_inches='tight', pad_inches=0.1)
        buf.seek(0)

        self._expanded_graph_image = Image.open(buf).copy()
        buf.close()

        # Reset zoom/pan
        self._expanded_zoom_level = 1.0
        self._expanded_pan_offset_x = 0
        self._expanded_pan_offset_y = 0

        self._update_expanded_display()

    def _draw_expanded_line_graph(self, graph_def):
        """Dessine un graphique en lignes dans l'expanded view"""
        axe_x = graph_def.get('x_axis', graph_def.get('axe_x', {}))
        axe_y = graph_def.get('y_axis', graph_def.get('axe_y', {}))
        courbes = graph_def.get('curves', graph_def.get('courbes', []))

        x_field = get_yaml_value(axe_x, 'field', 'champ', 'annee')
        x_key = FIELD_CODES.get(x_field, x_field)

        for courbe in courbes:
            y_field = get_yaml_value(courbe, 'field', 'champ', 'pib')
            y_key = FIELD_CODES.get(y_field, y_field)
            label = get_yaml_value(courbe, 'label', 'label', y_field)
            couleur = get_yaml_value(courbe, 'color', 'couleur', '#1f77b4')
            style = get_yaml_value(courbe, 'style', 'style', 'solid')

            diviseur = get_yaml_value(courbe, 'divisor', 'diviseur', 1)

            x_data = [r.get(x_key, r.get('annee', 0)) for r in self.resultats]
            y_data = [r.get(y_key, 0) / diviseur for r in self.resultats]

            linestyle = {'solid': '-', 'dashed': '--', 'dotted': ':'}
            ls = linestyle.get(style, '-')

            self.expanded_ax.plot(x_data, y_data, label=label, color=couleur, linestyle=ls, linewidth=2)

            if get_yaml_value(courbe, 'fill', 'remplir', False):
                alpha = get_yaml_value(courbe, 'fill_alpha', 'alpha_remplissage', 0.2)
                self.expanded_ax.fill_between(x_data, y_data, alpha=alpha, color=couleur)

        xlabel = get_yaml_value(axe_x, 'label', 'label', '').replace('{devise}', self.devise)
        ylabel = get_yaml_value(axe_y, 'label', 'label', '').replace('{devise}', self.devise)
        self.expanded_ax.set_xlabel(xlabel, fontsize=self.font_size)
        self.expanded_ax.set_ylabel(ylabel, fontsize=self.font_size)

        if axe_y.get('min') is not None:
            self.expanded_ax.set_ylim(bottom=axe_y['min'])

        if len(courbes) > 1:
            self.expanded_ax.legend(fontsize=self.font_size - 2)

    def _draw_expanded_bar_graph(self, graph_def, graph_id, stacked=False):
        """Dessine un graphique en barres dans l'expanded view"""
        import numpy as np
        salaires = self.params.salaires_analyses
        assurances_total = (self.params.assurance_sante + self.params.assurance_chomage +
                          self.params.assurance_pension + self.params.assurance_education)

        if 'an0' in graph_id or 'annee0' in graph_id.lower():
            annee_0 = next((r for r in self.resultats if r['annee'] == 1), None)
            if not annee_0:
                return
            diff_pct = annee_0['differentiel_pct'] / 100
        else:
            diff_pct = 0

        impacts, gains_ti, effets_nets, pourcentages = [], [], [], []

        for salaire in salaires:
            net_actuel = salaire * (1 - self.params.prelevement_actuel_total)
            base_imposable = max(0, salaire - self.params.abattement_forfaitaire)
            flat_tax = base_imposable * self.params.taux_flat_tax
            differentiel = salaire * diff_pct
            net_nouveau = salaire - flat_tax - differentiel - assurances_total

            impact = round(net_nouveau - net_actuel)
            impacts.append(impact)

            taux_ti = self.params.taxes_indirectes.get(salaire, 0.10)
            gain_ti = round(net_actuel * taux_ti)
            gains_ti.append(gain_ti)
            total = impact + gain_ti
            effets_nets.append(total)
            pct = (total / net_actuel) * 100 if net_actuel > 0 else 0
            pourcentages.append(pct)

        x = np.arange(len(salaires))
        width = 0.6
        labels = [f"{s}{self.devise}" for s in salaires]

        if stacked:
            self.expanded_ax.bar(x, impacts, width, color=COLORS['warning'], alpha=0.85,
                       label=f'Nouveau système* (ass. {assurances_total:.0f}{self.devise} incl.)')
            self.expanded_ax.bar(x, gains_ti, width, bottom=impacts, color=COLORS['success'],
                       alpha=0.85, label='Taxes indirectes abolies')

            for i, (imp, gti, total, pct) in enumerate(zip(impacts, gains_ti, effets_nets, pourcentages)):
                self.expanded_ax.annotate(f'+{imp}{self.devise}', (i, imp/2), ha='center', va='center',
                               fontsize=self.font_size, fontweight='bold')
                self.expanded_ax.annotate(f'+{gti}{self.devise}', (i, imp + gti/2), ha='center', va='center',
                               fontsize=self.font_size, fontweight='bold')
                self.expanded_ax.annotate(f'= +{total}{self.devise}\n(+{pct:.1f}%)', (i, total + 30), ha='center',
                               fontsize=self.font_size + 1, fontweight='bold', color=COLORS['dark'])
        else:
            self.expanded_ax.bar(x, effets_nets, width, color=COLORS['success'], alpha=0.85)
            for i, (total, pct) in enumerate(zip(effets_nets, pourcentages)):
                self.expanded_ax.annotate(f'+{total}{self.devise}\n(+{pct:.1f}%)', (i, total + 20), ha='center',
                               fontsize=self.font_size + 1, fontweight='bold')

        self.expanded_ax.axhline(y=0, color=COLORS['dark'], linewidth=1)
        self.expanded_ax.set_xticks(x)
        self.expanded_ax.set_xticklabels(labels, fontsize=self.font_size)
        self.expanded_ax.set_xlabel('Salaire brut', fontsize=self.font_size)
        self.expanded_ax.set_ylabel(f'Gain mensuel ({self.devise})', fontsize=self.font_size)

        if stacked:
            self.expanded_ax.legend(fontsize=self.font_size - 2, loc='upper left')

    def _draw_expanded_salary_evolution_graph(self, graph_def):
        """Dessine l'évolution du pouvoir d'achat dans l'expanded view"""
        axe_y = graph_def.get('y_axis', graph_def.get('axe_y', {}))
        mode = axe_y.get('mode', 'absolute')
        salaires_liste = graph_def.get('salaries', graph_def.get('salaires', self.params.salaires_analyses))
        show_baseline = graph_def.get('show_baseline', False)

        assurances_total = (self.params.assurance_sante + self.params.assurance_chomage +
                          self.params.assurance_pension + self.params.assurance_education)

        colors = ['#2ecc71', '#3498db', '#9b59b6', '#e74c3c', '#f39c12', '#1abc9c']

        # Récupérer le différentiel de l'année 1 (premier année de transition)
        # pour l'utiliser à l'année 0 si celle-ci n'a pas de différentiel
        diff_annee1 = 0
        for r in self.resultats:
            if r.get('annee', 0) == 1:
                diff_annee1 = r.get('differentiel_pct', 0) / 100
                break

        for idx, salaire in enumerate(salaires_liste):
            net_actuel = salaire * (1 - self.params.prelevement_actuel_total)
            taux_ti = self.params.taxes_indirectes.get(salaire, 0.10)
            gain_taxes = net_actuel * taux_ti

            annees = []
            valeurs = []

            for resultat in self.resultats:
                annee = resultat.get('annee', 0)
                diff_pct = resultat.get('differentiel_pct', 0) / 100

                # À l'année 0, utiliser le différentiel de l'année 1 pour cohérence
                # avec le graphique "Effet combiné année 0"
                if annee == 0 and diff_pct == 0 and diff_annee1 > 0:
                    diff_pct = diff_annee1

                base_imposable = max(0, salaire - self.params.abattement_forfaitaire)
                flat_tax = base_imposable * self.params.taux_flat_tax
                differentiel = salaire * diff_pct
                net_nouveau = salaire - flat_tax - differentiel - assurances_total + gain_taxes

                if mode == 'absolute':
                    valeur = net_nouveau
                elif mode == 'gain_absolute':
                    valeur = net_nouveau - net_actuel
                elif mode == 'gain_percent':
                    valeur = ((net_nouveau - net_actuel) / net_actuel * 100) if net_actuel > 0 else 0
                else:
                    valeur = net_nouveau

                annees.append(annee)
                valeurs.append(valeur)

            color = colors[idx % len(colors)]
            self.expanded_ax.plot(annees, valeurs, label=f"{salaire}{self.devise}", color=color, linewidth=2.5)

        # Aligner l'axe X pour que l'année 0 soit sur l'axe vertical
        if annees:
            self.expanded_ax.set_xlim(left=0)

        if show_baseline and mode == 'absolute':
            for idx, salaire in enumerate(salaires_liste):
                net_actuel = salaire * (1 - self.params.prelevement_actuel_total)
                color = colors[idx % len(colors)]
                self.expanded_ax.axhline(y=net_actuel, color=color, linestyle='--', alpha=0.3)

        xlabel = get_yaml_value(graph_def.get('x_axis', graph_def.get('axe_x', {})), 'label', 'label', 'Année')
        ylabel = get_yaml_value(axe_y, 'label', 'label', 'Salaire net').replace('{devise}', self.devise)
        self.expanded_ax.set_xlabel(xlabel, fontsize=self.font_size)
        self.expanded_ax.set_ylabel(ylabel, fontsize=self.font_size)
        self.expanded_ax.legend(fontsize=self.font_size - 2, loc='best')

    def _save_expanded_graph(self, graph_id):
        """Sauvegarde le graphique agrandi"""
        from tkinter import filedialog
        # Générer nom de fichier intelligent
        parts = []
        scenario = self._get_scenario_name()
        if scenario:
            parts.append(Path(scenario).stem)
        parts.append(graph_id)
        modified = self._get_modified_params()
        for param_id, value in modified.items():
            code = next((c for c, p in PARAM_CODES.items() if p == param_id), param_id[:4].upper())
            parts.append(f"{code}={value}")
        default_name = "_".join(parts)

        file_path = filedialog.asksaveasfilename(
            title=t('dialogs.save_graph_title'),
            defaultextension=".svg",
            initialfile=default_name,
            filetypes=[
                (t('file_types.svg'), "*.svg"),
                (t('file_types.png'), "*.png"),
                (t('file_types.pdf'), "*.pdf"),
                (t('file_types.jpeg'), "*.jpeg"),
                (t('file_types.all'), "*.*")
            ]
        )
        if file_path:
            try:
                self.expanded_fig.savefig(file_path, bbox_inches='tight', dpi=150)
                self._set_status(t('status.graph_saved', path=file_path))
            except Exception as e:
                messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_save', error=e))

    def _copy_expanded_graph_png(self):
        """Copie le graphique agrandi dans le presse-papier en PNG"""
        if not hasattr(self, 'expanded_fig') or self.expanded_fig is None:
            messagebox.showwarning(t('dialogs.warning'), t('dialogs.no_graph_to_copy'))
            return

        try:
            buf = io.BytesIO()
            self.expanded_fig.savefig(buf, format='png', dpi=150, bbox_inches='tight',
                                     facecolor='white', edgecolor='none')
            buf.seek(0)

            system = platform.system()
            if system == 'Linux':
                try:
                    subprocess.run(['which', 'xclip'], check=True, capture_output=True)
                except subprocess.CalledProcessError:
                    messagebox.showerror(t('dialogs.error'), t('dialogs.xclip_not_found'))
                    return
                process = subprocess.Popen(['xclip', '-selection', 'clipboard', '-t', 'image/png'],
                                         stdin=subprocess.PIPE)
                process.communicate(buf.getvalue())
            elif system == 'Darwin':
                import tempfile
                with tempfile.NamedTemporaryFile(suffix='.png', delete=False) as tmp:
                    tmp.write(buf.getvalue())
                    tmp_path = tmp.name
                subprocess.run(['osascript', '-e',
                              f'set the clipboard to (read (POSIX file "{tmp_path}") as «class PNGf»)'])
                os.unlink(tmp_path)
            elif system == 'Windows':
                try:
                    import win32clipboard
                    from PIL import Image
                    img = Image.open(buf)
                    output = io.BytesIO()
                    img.convert('RGB').save(output, 'BMP')
                    data = output.getvalue()[14:]
                    output.close()
                    win32clipboard.OpenClipboard()
                    win32clipboard.EmptyClipboard()
                    win32clipboard.SetClipboardData(win32clipboard.CF_DIB, data)
                    win32clipboard.CloseClipboard()
                except ImportError:
                    messagebox.showerror(t('dialogs.error'),
                        "Module pywin32 requis pour Windows.\nInstallez avec: pip install pywin32")
                    return

            self._set_status(t('status.graph_copied_png'))
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_copy', error=e))

    def _copy_expanded_graph_svg(self):
        """Copie le graphique agrandi dans le presse-papier en SVG (texte)"""
        if not hasattr(self, 'expanded_fig') or self.expanded_fig is None:
            messagebox.showwarning(t('dialogs.warning'), t('dialogs.no_graph_to_copy'))
            return

        try:
            buf = io.BytesIO()
            self.expanded_fig.savefig(buf, format='svg', bbox_inches='tight',
                                     facecolor='white', edgecolor='none')
            buf.seek(0)
            svg_content = buf.getvalue().decode('utf-8')

            self.clipboard_clear()
            self.clipboard_append(svg_content)
            self.update()

            self._set_status(t('status.graph_copied_svg'))
        except Exception as e:
            messagebox.showerror(t('dialogs.error'), t('dialogs.cannot_copy', error=e))

    def _minimize_expanded(self):
        """Réduit le graphique agrandi et revient aux 4 panneaux"""
        # Unbind Échap
        self.unbind('<Escape>')

        # Cacher l'expanded frame
        self.expanded_frame.pack_forget()

        # Nettoyer
        for widget in self.expanded_frame.winfo_children():
            widget.destroy()
        self.expanded_graph_panel = None
        self.expanded_source_index = None

        # Réafficher la grille des 4 graphiques
        self.graphs_frame.pack(fill='both', expand=True)

    def _on_edit_graph(self, file_path):
        EditorWindow(self, file_path, 'yaml', font_size=self.font_size, on_save=self._reload_graphs)

    def _create_new_graph(self):
        if not TEMPLATE_FILE.exists():
            messagebox.showerror(t('dialogs.error'), t('dialogs.template_not_found'))
            return

        import tkinter.simpledialog
        name = tkinter.simpledialog.askstring(t('dialogs.new_graph_title'), t('dialogs.new_graph_prompt'), parent=self)

        if name:
            new_file = GRAPHIQUES_DIR / f"{name}.yaml"
            if new_file.exists():
                messagebox.showerror(t('dialogs.error'), t('dialogs.file_exists', name=name))
                return

            shutil.copy(TEMPLATE_FILE, new_file)
            self._reload_graphs()
            EditorWindow(self, new_file, 'yaml', font_size=self.font_size, on_save=self._reload_graphs)

    def _reload_graphs(self):
        self.graph_files = load_graph_files()
        for panel in self.graph_panels:
            panel.update_graph_files(self.graph_files)
            panel.redraw()

    def _open_graphs_folder(self):
        import subprocess
        if sys.platform == 'win32':
            os.startfile(GRAPHIQUES_DIR)
        elif sys.platform == 'darwin':
            subprocess.run(['open', GRAPHIQUES_DIR])
        else:
            subprocess.run(['xdg-open', GRAPHIQUES_DIR])

    def _edit_current_scenario(self):
        """Bascule l'éditeur de scénario dans le panneau paramètres"""
        scenario_rel = self.scenario_var.get()
        if scenario_rel:
            path = CONFIG_DIR / scenario_rel
            self.param_panel.set_scenario_file(path)
            self.param_panel._on_edit_scenario_click()
        else:
            self._set_status(t('dialogs.no_scenario_loaded'))

    def _reload_current_scenario(self):
        """Recharge le scénario actuel après édition"""
        scenario_rel = self.scenario_var.get()
        if scenario_rel:
            scenario_path = CONFIG_DIR / scenario_rel
            self._load_scenario(scenario_path)

    def _get_modified_params(self):
        """Retourne les paramètres modifiés (callback pour GraphPanel)"""
        return self.param_panel.get_modified_params_dict()

    def _get_scenario_name(self):
        """Retourne le nom du scénario actuel (callback pour GraphPanel)"""
        return self.scenario_var.get()

    def _change_font_size(self, new_size):
        """Change la taille de police de l'application entière (GUI, menus, graphiques, dialogues)"""
        # Limiter la taille entre 10 et 30
        new_size = max(10, min(30, new_size))
        self.font_size = new_size

        # Appliquer le scaling global (polices, dialogues, widgets)
        self._apply_scaling()

        # Note: la fenêtre ne change plus de taille quand on modifie la police

        # Mettre à jour le panneau des paramètres
        self.param_panel.update_font_size(new_size)

        # Mettre à jour le texte d'aide
        self.help_text.configure(state='normal')
        self.help_text.configure(font=('Consolas', new_size - 1))
        self.help_text.configure(state='disabled')

        # Redessiner tous les graphiques avec les nouvelles polices
        for panel in self.graph_panels:
            panel.update_font_size(new_size)
            panel.redraw()

        # Mettre à jour les fenêtres Live
        for window in self.enlarged_windows[:]:
            try:
                if window.winfo_exists():
                    window.update_font_size(new_size)
            except:
                pass

        # Mettre à jour le titre
        self._update_title()

        # Mettre à jour le menu Affichage pour refléter la nouvelle taille
        self._update_view_menu()

    def _change_language(self, lang_code):
        """Change la langue de l'interface"""
        if load_language(lang_code):
            # Recréer les menus avec les nouveaux libellés
            self._create_menu()
            # Recréer les widgets du panneau gauche
            self._refresh_ui_labels()
            # Mettre à jour le titre
            self._update_title()

    def _refresh_ui_labels(self):
        """Met à jour tous les labels de l'interface avec la langue courante"""
        # Mettre à jour les LabelFrames
        self.help_frame.configure(text=f"📖 {t('panel.help')}")
        # Mettre à jour le menu scénario
        self._update_scenario_menu()
        # Mettre à jour les labels des sections du panneau paramètres
        self.param_panel.refresh_labels()
        # Recréer les pannels graphiques avec les nouveaux labels
        for panel in self.graph_panels:
            panel.refresh_labels()

    def _show_about(self):
        font_factor = self.font_size / 12.0
        current_scaling = self.tk.call('tk', 'scaling')
        messagebox.showinfo(
            t('about.title'),
            f"{t('app.title')} {t('app.version')}\n\n"
            f"{t('about.new_features')}\n"
            f"• 💾 {t('about.feature_save')}\n"
            f"• {t('about.feature_filenames')}\n"
            f"• {t('about.feature_param_codes')}\n"
            f"• {t('about.feature_scaling')}\n"
            f"• {t('about.feature_visual_editor')}\n\n"
            f"{t('about.features')}\n"
            f"• {t('about.feature_auto_sim')}\n"
            f"• {t('about.feature_readonly')}\n"
            f"• {t('about.feature_auto_update')}\n"
            f"• {t('about.feature_colors')}\n"
            f"• {t('about.feature_modified')}\n"
            f"• {t('about.feature_live')}\n"
            f"• {t('about.feature_yaml_editor')}\n\n"
            f"Police: {self.font_size}pt | Factor: x{font_factor:.1f}\n"
            f"Tk scaling: {current_scaling:.2f}\n"
            f"{t('about.footer')}"
        )


def main():
    app = SimulateurApp()
    app.mainloop()


if __name__ == "__main__":
    main()
