#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Registre des métriques pour le simulateur de transition des pensions.

Ce module définit toutes les métriques disponibles, leur classification
(normalisée, indexée, absolue) et les fonctions de calcul des métriques dérivées.
"""

from dataclasses import dataclass
from typing import List, Dict, Optional, Callable
from enum import Enum


class MetricCategory(Enum):
    """Catégorie de métrique déterminant le comportement de l'axe Y"""
    NORMALIZED = "normalized"  # % du PIB → axe Y commun
    INDEXED = "indexed"        # Base 100 → axe Y commun
    ABSOLUTE = "absolute"      # Valeur brute → axe Y indépendant
    COUNT = "count"            # Compteur (nombre) → axe Y indépendant


@dataclass
class MetricDefinition:
    """Définition complète d'une métrique"""
    code: str                    # Code 4 lettres (ex: 'DPPC')
    field: str                   # Nom du champ dans resultats (ex: 'dette_publique_pct')
    label_fr: str                # Nom affiché en français
    label_en: str                # Nom affiché en anglais
    category: MetricCategory     # Type de métrique
    unit: str                    # Unité (%, Mds, index, nombre)
    derived_from: Optional[str]  # Si calculé: champ source. Sinon: None
    description_fr: str = ""     # Description détaillée
    description_en: str = ""     # Description détaillée en anglais

    def get_label(self, lang: str = 'fr') -> str:
        """Retourne le label dans la langue demandée"""
        return self.label_fr if lang == 'fr' else self.label_en

    def get_description(self, lang: str = 'fr') -> str:
        """Retourne la description dans la langue demandée"""
        return self.description_fr if lang == 'fr' else self.description_en

    def is_common_axis(self) -> bool:
        """Indique si cette métrique partage un axe Y commun"""
        return self.category in (MetricCategory.NORMALIZED, MetricCategory.INDEXED)


# =============================================================================
# REGISTRE DES MÉTRIQUES
# =============================================================================

METRICS_REGISTRY: Dict[str, MetricDefinition] = {
    # -------------------------------------------------------------------------
    # MÉTRIQUES NORMALISÉES (% du PIB) - Axe Y commun
    # -------------------------------------------------------------------------
    'DPPC': MetricDefinition(
        code='DPPC',
        field='dette_publique_pct',
        label_fr='Dette publique (% PIB)',
        label_en='Public Debt (% GDP)',
        category=MetricCategory.NORMALIZED,
        unit='%',
        derived_from=None,
        description_fr='Dette publique en pourcentage du PIB',
        description_en='Public debt as percentage of GDP'
    ),
    'DTPC': MetricDefinition(
        code='DTPC',
        field='dette_transition_pct',
        label_fr='Dette transition (% PIB)',
        label_en='Transition Debt (% GDP)',
        category=MetricCategory.NORMALIZED,
        unit='%',
        derived_from=None,
        description_fr='Dette de transition en pourcentage du PIB',
        description_en='Transition debt as percentage of GDP'
    ),
    'TTPC': MetricDefinition(
        code='TTPC',
        field='dette_publique_totale_pct',
        label_fr='Dette totale (% PIB)',
        label_en='Total Debt (% GDP)',
        category=MetricCategory.NORMALIZED,
        unit='%',
        derived_from=None,
        description_fr='Dette publique totale en pourcentage du PIB',
        description_en='Total public debt as percentage of GDP'
    ),
    'DIPC': MetricDefinition(
        code='DIPC',
        field='dette_implicite_pct',
        label_fr='Dette implicite (% PIB)',
        label_en='Implicit Debt (% GDP)',
        category=MetricCategory.NORMALIZED,
        unit='%',
        derived_from=None,
        description_fr='Dette implicite des pensions en pourcentage du PIB',
        description_en='Implicit pension debt as percentage of GDP'
    ),
    'DIFF': MetricDefinition(
        code='DIFF',
        field='differentiel_pct',
        label_fr='Différentiel (% PIB)',
        label_en='Differential (% GDP)',
        category=MetricCategory.NORMALIZED,
        unit='%',
        derived_from=None,
        description_fr='Différentiel fiscal en pourcentage du PIB',
        description_en='Tax differential as percentage of GDP'
    ),
    'DFPL': MetricDefinition(
        code='DFPL',
        field='differentiel_plafond_pct',
        label_fr='Plafond différentiel (% PIB)',
        label_en='Differential Cap (% GDP)',
        category=MetricCategory.NORMALIZED,
        unit='%',
        derived_from=None,
        description_fr='Plafond constitutionnel du différentiel',
        description_en='Constitutional cap on differential'
    ),
    'SBPC': MetricDefinition(
        code='SBPC',
        field='surplus_budgetaire_pct',
        label_fr='Surplus budgétaire (% PIB)',
        label_en='Budget Surplus (% GDP)',
        category=MetricCategory.NORMALIZED,
        unit='%',
        derived_from=None,
        description_fr='Surplus budgétaire minimal en pourcentage du PIB',
        description_en='Minimum budget surplus as percentage of GDP'
    ),

    # -------------------------------------------------------------------------
    # MÉTRIQUES INDEXÉES (Base 100) - Axe Y commun
    # -------------------------------------------------------------------------
    'PIBI': MetricDefinition(
        code='PIBI',
        field='pib_indexe',
        label_fr='PIB indexé (base 100)',
        label_en='Indexed GDP (base 100)',
        category=MetricCategory.INDEXED,
        unit='index',
        derived_from='pib',
        description_fr='PIB indexé, base 100 à l\'année 0',
        description_en='Indexed GDP, base 100 at year 0'
    ),
    'DPBI': MetricDefinition(
        code='DPBI',
        field='dette_publique_indexee',
        label_fr='Dette pub. indexée (base 100)',
        label_en='Indexed Public Debt (base 100)',
        category=MetricCategory.INDEXED,
        unit='index',
        derived_from='dette_publique',
        description_fr='Dette publique indexée, base 100 à l\'année 0',
        description_en='Indexed public debt, base 100 at year 0'
    ),
    'DTBI': MetricDefinition(
        code='DTBI',
        field='dette_totale_indexee',
        label_fr='Dette totale indexée (base 100)',
        label_en='Indexed Total Debt (base 100)',
        category=MetricCategory.INDEXED,
        unit='index',
        derived_from='dette_publique_totale',
        description_fr='Dette totale indexée, base 100 à l\'année 0',
        description_en='Indexed total debt, base 100 at year 0'
    ),
    'DIBI': MetricDefinition(
        code='DIBI',
        field='dette_implicite_indexee',
        label_fr='Dette impl. indexée (base 100)',
        label_en='Indexed Implicit Debt (base 100)',
        category=MetricCategory.INDEXED,
        unit='index',
        derived_from='dette_implicite',
        description_fr='Dette implicite indexée, base 100 à l\'année 0',
        description_en='Indexed implicit debt, base 100 at year 0'
    ),
    'INBI': MetricDefinition(
        code='INBI',
        field='interets_indexes',
        label_fr='Intérêts indexés (base 100)',
        label_en='Indexed Interest (base 100)',
        category=MetricCategory.INDEXED,
        unit='index',
        derived_from='interets_totaux',
        description_fr='Intérêts indexés, base 100 à l\'année 1',
        description_en='Indexed interest, base 100 at year 1'
    ),

    # -------------------------------------------------------------------------
    # MÉTRIQUES ABSOLUES (Mds devise) - Axe Y indépendant
    # -------------------------------------------------------------------------
    'PIBS': MetricDefinition(
        code='PIBS',
        field='pib',
        label_fr='PIB (Mds)',
        label_en='GDP (Bn)',
        category=MetricCategory.ABSOLUTE,
        unit='Mds',
        derived_from=None,
        description_fr='Produit Intérieur Brut en milliards',
        description_en='Gross Domestic Product in billions'
    ),
    'DPUB': MetricDefinition(
        code='DPUB',
        field='dette_publique',
        label_fr='Dette publique (Mds)',
        label_en='Public Debt (Bn)',
        category=MetricCategory.ABSOLUTE,
        unit='Mds',
        derived_from=None,
        description_fr='Dette publique en milliards',
        description_en='Public debt in billions'
    ),
    'DTRA': MetricDefinition(
        code='DTRA',
        field='dette_transition',
        label_fr='Dette transition (Mds)',
        label_en='Transition Debt (Bn)',
        category=MetricCategory.ABSOLUTE,
        unit='Mds',
        derived_from=None,
        description_fr='Dette de transition en milliards',
        description_en='Transition debt in billions'
    ),
    'DTOT': MetricDefinition(
        code='DTOT',
        field='dette_publique_totale',
        label_fr='Dette totale (Mds)',
        label_en='Total Debt (Bn)',
        category=MetricCategory.ABSOLUTE,
        unit='Mds',
        derived_from=None,
        description_fr='Dette publique totale en milliards',
        description_en='Total public debt in billions'
    ),
    'DIMP': MetricDefinition(
        code='DIMP',
        field='dette_implicite',
        label_fr='Dette implicite (Mds)',
        label_en='Implicit Debt (Bn)',
        category=MetricCategory.ABSOLUTE,
        unit='Mds',
        derived_from=None,
        description_fr='Dette implicite des pensions en milliards',
        description_en='Implicit pension debt in billions'
    ),
    'FLUX': MetricDefinition(
        code='FLUX',
        field='flux_pensions',
        label_fr='Flux pensions (Mds/an)',
        label_en='Pension Flow (Bn/yr)',
        category=MetricCategory.ABSOLUTE,
        unit='Mds',
        derived_from=None,
        description_fr='Flux annuel des pensions en milliards',
        description_en='Annual pension flow in billions'
    ),
    'INTR': MetricDefinition(
        code='INTR',
        field='interets_totaux',
        label_fr='Intérêts (Mds/an)',
        label_en='Interest (Bn/yr)',
        category=MetricCategory.ABSOLUTE,
        unit='Mds',
        derived_from=None,
        description_fr='Intérêts annuels sur les dettes en milliards',
        description_en='Annual interest on debts in billions'
    ),
    'SURP': MetricDefinition(
        code='SURP',
        field='surplus_budgetaire',
        label_fr='Surplus budgétaire (Mds)',
        label_en='Budget Surplus (Bn)',
        category=MetricCategory.ABSOLUTE,
        unit='Mds',
        derived_from=None,
        description_fr='Surplus budgétaire en milliards',
        description_en='Budget surplus in billions'
    ),

    # -------------------------------------------------------------------------
    # COMPTEURS - Axe Y indépendant
    # -------------------------------------------------------------------------
    'ANNE': MetricDefinition(
        code='ANNE',
        field='annee',
        label_fr='Année',
        label_en='Year',
        category=MetricCategory.COUNT,
        unit='',
        derived_from=None,
        description_fr='Année de simulation',
        description_en='Simulation year'
    ),
    'NRET': MetricDefinition(
        code='NRET',
        field='nombre_retraites',
        label_fr='Nombre de retraités',
        label_en='Number of Retirees',
        category=MetricCategory.COUNT,
        unit='',
        derived_from=None,
        description_fr='Nombre total de retraités',
        description_en='Total number of retirees'
    ),
    'REQR': MetricDefinition(
        code='REQR',
        field='retraites_equiv_repartition',
        label_fr='Équiv. retraités (pondéré)',
        label_en='Equiv. Retirees (weighted)',
        category=MetricCategory.COUNT,
        unit='',
        derived_from=None,
        description_fr='Nombre de retraités pondéré par les droits à répartition',
        description_en='Number of retirees weighted by distribution rights'
    ),
}

# Mapping inverse: field -> code
FIELD_TO_CODE: Dict[str, str] = {m.field: m.code for m in METRICS_REGISTRY.values()}


# =============================================================================
# FONCTIONS DE CALCUL DES MÉTRIQUES DÉRIVÉES
# =============================================================================

def calculer_metrique_indexee(
    resultats: List[Dict],
    champ_source: str,
    champ_cible: str,
    annee_base: int = 0
) -> List[Dict]:
    """
    Transforme une métrique absolue en métrique indexée base 100.

    Args:
        resultats: Liste des résultats année par année
        champ_source: Nom du champ à indexer (ex: 'pib')
        champ_cible: Nom du nouveau champ (ex: 'pib_indexe')
        annee_base: Année de référence (base 100), défaut = 0

    Returns:
        Liste des résultats avec le nouveau champ ajouté
    """
    if not resultats:
        return resultats

    # Trouver la valeur de base
    valeur_base = None
    for r in resultats:
        if r.get('annee', -1) == annee_base:
            valeur_base = r.get(champ_source, 0)
            break

    # Si pas trouvé à l'année spécifiée, prendre la première valeur non nulle
    if valeur_base is None or valeur_base == 0:
        for r in resultats:
            v = r.get(champ_source, 0)
            if v != 0:
                valeur_base = v
                break

    # Calculer l'index pour chaque année
    for r in resultats:
        valeur = r.get(champ_source, 0)
        if valeur_base and valeur_base != 0:
            r[champ_cible] = (valeur / valeur_base) * 100
        else:
            r[champ_cible] = 0

    return resultats


def calculer_toutes_metriques_indexees(resultats: List[Dict]) -> List[Dict]:
    """
    Calcule toutes les métriques indexées à partir des métriques absolues.

    Args:
        resultats: Liste des résultats année par année

    Returns:
        Liste des résultats avec toutes les métriques indexées ajoutées
    """
    # Liste des métriques indexées à calculer
    metriques_a_calculer = [
        ('pib', 'pib_indexe', 0),
        ('dette_publique', 'dette_publique_indexee', 0),
        ('dette_publique_totale', 'dette_totale_indexee', 0),
        ('dette_implicite', 'dette_implicite_indexee', 0),
        ('interets_totaux', 'interets_indexes', 1),  # Base à l'année 1 (pas 0 car souvent 0)
    ]

    for champ_source, champ_cible, annee_base in metriques_a_calculer:
        resultats = calculer_metrique_indexee(resultats, champ_source, champ_cible, annee_base)

    return resultats


# =============================================================================
# FONCTIONS UTILITAIRES
# =============================================================================

def get_metric(code: str) -> Optional[MetricDefinition]:
    """Retourne la définition d'une métrique par son code"""
    return METRICS_REGISTRY.get(code.upper())


def get_metric_by_field(field: str) -> Optional[MetricDefinition]:
    """Retourne la définition d'une métrique par son nom de champ"""
    code = FIELD_TO_CODE.get(field)
    return METRICS_REGISTRY.get(code) if code else None


def get_metrics_by_category(category: MetricCategory) -> List[MetricDefinition]:
    """Retourne toutes les métriques d'une catégorie"""
    return [m for m in METRICS_REGISTRY.values() if m.category == category]


def get_normalized_metrics() -> List[MetricDefinition]:
    """Retourne les métriques normalisées (axe Y commun)"""
    return get_metrics_by_category(MetricCategory.NORMALIZED)


def get_indexed_metrics() -> List[MetricDefinition]:
    """Retourne les métriques indexées (axe Y commun)"""
    return get_metrics_by_category(MetricCategory.INDEXED)


def get_absolute_metrics() -> List[MetricDefinition]:
    """Retourne les métriques absolues (axe Y indépendant)"""
    return get_metrics_by_category(MetricCategory.ABSOLUTE)


def get_common_axis_metrics() -> List[MetricDefinition]:
    """Retourne toutes les métriques à axe Y commun"""
    return [m for m in METRICS_REGISTRY.values() if m.is_common_axis()]


def get_metrics_for_comparison() -> Dict[str, List[MetricDefinition]]:
    """
    Retourne les métriques groupées pour le dropdown de comparaison.

    Returns:
        Dict avec les clés 'normalized', 'indexed', 'absolute'
    """
    return {
        'normalized': get_normalized_metrics(),
        'indexed': get_indexed_metrics(),
        'absolute': get_absolute_metrics(),
    }


def resolve_field_code(code_or_name: str) -> str:
    """
    Convertit un code 4 lettres en nom de champ, ou retourne le nom tel quel.

    Args:
        code_or_name: Code 4 lettres (ex: 'DPPC') ou nom de champ

    Returns:
        Nom du champ correspondant
    """
    if not code_or_name:
        return code_or_name

    metric = METRICS_REGISTRY.get(code_or_name.upper())
    if metric:
        return metric.field
    return code_or_name
