#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Mode Comparaison pour le simulateur de transition des pensions.

Ce module implémente la comparaison visuelle de plusieurs scénarios via
une grille de graphiques juxtaposés :
- Colonnes = scénarios (pays)
- Lignes = graphiques (les mêmes graphiques YAML qu'en mode normal)

Chaque ligne affiche le MÊME graphique pour tous les pays sélectionnés.
"""

import tkinter as tk
from tkinter import ttk
from dataclasses import dataclass, field
from typing import List, Dict, Optional, Callable, Any, Tuple
from pathlib import Path
import configparser

from metrics_registry import calculer_toutes_metriques_indexees


# =============================================================================
# CONSTANTES
# =============================================================================

TOOLTIP_DELAY_MS = 500


# =============================================================================
# STRUCTURES DE DONNÉES
# =============================================================================

@dataclass
class ComparisonConfig:
    """Configuration du mode comparaison"""
    scenarios: List[str] = field(default_factory=list)  # Chemins .ini
    graphs: List[str] = field(default_factory=list)     # Noms des graphiques YAML

    MIN_COLS: int = 2
    MAX_COLS: int = 4
    MIN_ROWS: int = 1
    MAX_ROWS: int = 4

    def can_add_scenario(self) -> bool:
        return len(self.scenarios) < self.MAX_COLS

    def can_add_graph(self) -> bool:
        return len(self.graphs) < self.MAX_ROWS

    def can_remove_scenario(self) -> bool:
        return len(self.scenarios) > self.MIN_COLS

    def can_remove_graph(self) -> bool:
        return len(self.graphs) > self.MIN_ROWS


@dataclass
class ScenarioData:
    """Données d'un scénario chargé"""
    name: str
    ini_path: str
    params: Any
    results: List[Dict]
    devise: str = "€"


# =============================================================================
# FONCTIONS UTILITAIRES
# =============================================================================

def get_available_scenarios(config_dir: Path) -> List[Tuple[str, str]]:
    """
    Retourne la liste des scénarios disponibles (recherche récursive).

    Returns:
        Liste de tuples (chemin_ini, nom_affiché)
    """
    scenarios = []

    if not config_dir.exists():
        return scenarios

    # Recherche RÉCURSIVE pour trouver les .ini dans les sous-dossiers
    for ini_file in sorted(config_dir.rglob("*.ini")):
        try:
            config = configparser.ConfigParser()
            config.read(ini_file, encoding='utf-8')
            name = config.get('Pays', 'nom', fallback=ini_file.stem)
            scenarios.append((str(ini_file), name))
        except Exception:
            scenarios.append((str(ini_file), ini_file.stem))

    return scenarios


# =============================================================================
# WIDGET: TOOLTIP AVEC DÉLAI
# =============================================================================

class DelayedTooltip:
    """Tooltip qui apparaît après un délai"""

    def __init__(self, widget: tk.Widget, text: str, delay_ms: int = TOOLTIP_DELAY_MS):
        self.widget = widget
        self.text = text
        self.delay_ms = delay_ms
        self.tooltip_window = None
        self.schedule_id = None

        widget.bind('<Enter>', self._schedule_show)
        widget.bind('<Leave>', self._hide)
        widget.bind('<Button-1>', self._hide)

    def _schedule_show(self, event):
        self._cancel_scheduled()
        self.schedule_id = self.widget.after(self.delay_ms, lambda: self._show(event))

    def _cancel_scheduled(self):
        if self.schedule_id:
            self.widget.after_cancel(self.schedule_id)
            self.schedule_id = None

    def _show(self, event):
        if self.tooltip_window:
            return
        x = event.x_root + 10
        y = event.y_root + 10
        self.tooltip_window = tk.Toplevel(self.widget)
        self.tooltip_window.wm_overrideredirect(True)
        self.tooltip_window.wm_geometry(f"+{x}+{y}")
        label = tk.Label(self.tooltip_window, text=self.text, background="#fffde7",
                        foreground="#333", relief='solid', borderwidth=1, padx=6, pady=3)
        label.pack()

    def _hide(self, event=None):
        self._cancel_scheduled()
        if self.tooltip_window:
            self.tooltip_window.destroy()
            self.tooltip_window = None

    def update_text(self, new_text: str):
        self.text = new_text


# =============================================================================
# PANNEAU PRINCIPAL : GRILLE DE COMPARAISON
# =============================================================================

class ComparisonGridPanel(ttk.Frame):
    """
    Panneau principal du mode comparaison.
    Design compact avec icônes ▼ et boutons [×] / [+].
    """

    def __init__(self, parent,
                 config: ComparisonConfig,
                 available_scenarios: List[Tuple[str, str]],
                 graph_files: Dict[str, Any],
                 load_scenario_callback: Callable[[str], Tuple[Any, List[Dict], str]],
                 GraphPanelClass,
                 font_size: int = 12,
                 on_status_update: Optional[Callable[[str], None]] = None,
                 add_tooltip_callback: Optional[Callable] = None,
                 on_switch_to_normal: Optional[Callable[[str], None]] = None):
        super().__init__(parent)

        self.config = config
        self.available_scenarios = available_scenarios
        self.graph_files = graph_files
        self.load_scenario_callback = load_scenario_callback
        self.GraphPanelClass = GraphPanelClass
        self.font_size = font_size
        self.on_status_update = on_status_update
        self.add_tooltip = add_tooltip_callback or (lambda w, t: None)
        self.on_switch_to_normal = on_switch_to_normal

        self.scenarios_data: Dict[str, ScenarioData] = {}
        self.graph_panels: Dict[Tuple[int, int], Any] = {}

        self._create_widgets()
        self._load_initial_scenarios()

    def _set_status(self, message: str):
        if self.on_status_update:
            self.on_status_update(message)

    def _create_widgets(self):
        self.grid_frame = ttk.Frame(self)
        self.grid_frame.pack(fill='both', expand=True, padx=2, pady=2)
        self._rebuild_grid()

    def _rebuild_grid(self):
        """Reconstruit la grille avec design compact"""
        # Sauvegarder les graphiques par position
        current_graphs = {}
        for (row, col), panel in self.graph_panels.items():
            if hasattr(panel, 'current_graph_id'):
                current_graphs[(row, col)] = panel.current_graph_id

        # Nettoyer
        for widget in self.grid_frame.winfo_children():
            widget.destroy()
        self.graph_panels.clear()

        n_cols = len(self.config.scenarios)
        n_rows = len(self.config.graphs)

        # Configurer la grille
        for i in range(6):
            self.grid_frame.columnconfigure(i, weight=0)
            self.grid_frame.rowconfigure(i, weight=0)

        # Colonne 0 = sélecteurs de graphiques (compacte)
        self.grid_frame.columnconfigure(0, weight=0, minsize=50)
        for col in range(n_cols):
            self.grid_frame.columnconfigure(col + 1, weight=1)
        for row in range(n_rows):
            self.grid_frame.rowconfigure(row + 1, weight=1)

        # === LIGNE 0 : EN-TÊTE SCÉNARIOS ===
        # Cellule (0,0) vide
        ttk.Label(self.grid_frame, text="").grid(row=0, column=0, sticky='nsew')

        for col_idx, ini_path in enumerate(self.config.scenarios):
            scenario_data = self.scenarios_data.get(ini_path)
            name = scenario_data.name if scenario_data else Path(ini_path).stem

            header_frame = ttk.Frame(self.grid_frame)
            header_frame.grid(row=0, column=col_idx + 1, sticky='ew', padx=1, pady=1)

            # Bouton ▼ avec nom du scénario
            btn = ttk.Button(header_frame, text=f"{name} ▼", width=12,
                           command=lambda idx=col_idx: self._show_scenario_menu(idx))
            btn.pack(side='left', fill='x', expand=True)
            DelayedTooltip(btn, f"Cliquer pour changer de pays")

            # Bouton [×] pour supprimer ce scénario
            if self.config.can_remove_scenario():
                rem_btn = ttk.Button(header_frame, text="×", width=2,
                                    command=lambda idx=col_idx: self._remove_scenario_at(idx))
                rem_btn.pack(side='left', padx=1)

        # Bouton [+] pour ajouter un scénario (à droite de l'en-tête)
        if self.config.can_add_scenario():
            add_btn = ttk.Button(self.grid_frame, text="+", width=2, command=self._add_scenario)
            add_btn.grid(row=0, column=n_cols + 1, sticky='w', padx=1, pady=1)
            self.add_tooltip(add_btn, "Ajouter un pays")

        # === LIGNES DE GRAPHIQUES ===
        for row_idx, graph_id in enumerate(self.config.graphs):
            is_last_row = (row_idx == n_rows - 1)

            # Colonne 0 : sélecteur de graphique
            selector_frame = ttk.Frame(self.grid_frame)
            selector_frame.grid(row=row_idx + 1, column=0, sticky='nsew', padx=1, pady=1)

            # Bouton ▼ pour sélectionner le graphique
            graph_btn = ttk.Button(selector_frame, text="▼", width=3,
                                  command=lambda idx=row_idx: self._show_graph_menu(idx))
            graph_btn.pack(side='top', pady=1)

            # Tooltip avec le nom du graphique
            graph_name = self.graph_files.get(graph_id, {}).get('title', graph_id) if isinstance(self.graph_files.get(graph_id), dict) else graph_id
            DelayedTooltip(graph_btn, graph_name)

            # Bouton [×] pour supprimer cette ligne
            if self.config.can_remove_graph():
                rem_btn = ttk.Button(selector_frame, text="×", width=2,
                                    command=lambda idx=row_idx: self._remove_graph_at(idx))
                rem_btn.pack(side='top', pady=1)

            # Bouton [+] pour ajouter une ligne (uniquement sur la dernière ligne)
            if is_last_row and self.config.can_add_graph():
                add_btn = ttk.Button(selector_frame, text="+", width=2, command=self._add_graph)
                add_btn.pack(side='bottom', pady=1)
                self.add_tooltip(add_btn, "Ajouter un graphique")

            # Cellules graphiques pour chaque scénario
            for col_idx, ini_path in enumerate(self.config.scenarios):
                scenario_data = self.scenarios_data.get(ini_path)

                panel = self.GraphPanelClass(
                    self.grid_frame,
                    on_click_enlarge=lambda idx: None,
                    on_edit_graph=lambda idx: None,
                    panel_index=row_idx * n_cols + col_idx,
                    font_size=self.font_size,
                    on_status_update=self._set_status,
                    read_only=True  # Mode lecture seule en comparaison
                )
                panel.grid(row=row_idx + 1, column=col_idx + 1, sticky='nsew', padx=1, pady=1)
                panel.update_graph_files(self.graph_files)

                # Restaurer ou définir le graphique
                if (row_idx, col_idx) in current_graphs:
                    panel.set_graph(current_graphs[(row_idx, col_idx)])
                else:
                    panel.set_graph(graph_id)

                # Données du scénario
                if scenario_data:
                    panel.set_data(scenario_data.results, scenario_data.params, scenario_data.devise)

                self.graph_panels[(row_idx, col_idx)] = panel

        self._update_status()
        # Forcer un redraw après que le layout tkinter soit finalisé
        # (résout le problème de canvas non dimensionné au premier affichage)
        self.after(100, self._force_redraw_all_panels)
        # Synchroniser les échelles Y si même pays (après le redraw)
        self.after(200, self._synchronize_y_scales_if_same_country)

    def _show_scenario_menu(self, col_idx: int):
        """Affiche le menu de sélection de scénario"""
        menu = tk.Menu(self, tearoff=0)

        current_ini = self.config.scenarios[col_idx]

        # Option pour passer en mode normal avec ce scénario
        if self.on_switch_to_normal:
            menu.add_command(
                label="📊 Ouvrir en mode normal",
                command=lambda: self.on_switch_to_normal(current_ini)
            )
            menu.add_separator()

        # Liste des scénarios disponibles
        for ini_path, name in self.available_scenarios:
            is_selected = (ini_path == current_ini)
            menu.add_command(
                label=f"{'✓ ' if is_selected else '   '}{name}",
                command=lambda p=ini_path, n=name: self._select_scenario(col_idx, p)
            )

        # Afficher le menu
        widget = self.grid_frame.winfo_children()[col_idx + 1]  # header_frame
        menu.tk_popup(widget.winfo_rootx(), widget.winfo_rooty() + widget.winfo_height())

    def _select_scenario(self, col_idx: int, ini_path: str):
        """Sélectionne un scénario pour une colonne"""
        if ini_path == self.config.scenarios[col_idx]:
            return

        self.config.scenarios[col_idx] = ini_path
        self._load_scenario(ini_path)
        self._rebuild_grid()

    def _show_graph_menu(self, row_idx: int):
        """Affiche le menu de sélection de graphique"""
        menu = tk.Menu(self, tearoff=0)

        current_graph = self.config.graphs[row_idx]
        for graph_id, graph_def in self.graph_files.items():
            is_selected = (graph_id == current_graph)
            title = graph_def.get('title', graph_id) if isinstance(graph_def, dict) else graph_id
            menu.add_command(
                label=f"{'✓ ' if is_selected else '   '}{title}",
                command=lambda g=graph_id: self._select_graph(row_idx, g)
            )

        # Afficher le menu
        # Trouver le bon widget
        for widget in self.grid_frame.winfo_children():
            info = widget.grid_info()
            if info.get('row') == row_idx + 1 and info.get('column') == 0:
                menu.tk_popup(widget.winfo_rootx() + widget.winfo_width(), widget.winfo_rooty())
                break

    def _select_graph(self, row_idx: int, graph_id: str):
        """Sélectionne un graphique pour une ligne"""
        if graph_id == self.config.graphs[row_idx]:
            return

        self.config.graphs[row_idx] = graph_id

        # Mettre à jour tous les panneaux de cette ligne
        for col_idx in range(len(self.config.scenarios)):
            panel = self.graph_panels.get((row_idx, col_idx))
            if panel:
                panel.set_graph(graph_id)

        self._rebuild_grid()

    def _load_initial_scenarios(self):
        for ini_path in self.config.scenarios:
            self._load_scenario(ini_path)
        self._update_all_panels()

    def _load_scenario(self, ini_path: str) -> bool:
        if ini_path in self.scenarios_data:
            return True

        try:
            self._set_status(f"Chargement de {Path(ini_path).stem}...")
            params, results, devise = self.load_scenario_callback(ini_path)
            results = calculer_toutes_metriques_indexees(results)

            name = params.nom_pays if hasattr(params, 'nom_pays') else Path(ini_path).stem
            self.scenarios_data[ini_path] = ScenarioData(
                name=name, ini_path=ini_path, params=params, results=results, devise=devise
            )
            self._set_status(f"{name} chargé")
            return True
        except Exception as e:
            self._set_status(f"Erreur: {e}")
            return False

    def _update_all_panels(self):
        for (row_idx, col_idx), panel in self.graph_panels.items():
            if col_idx < len(self.config.scenarios):
                ini_path = self.config.scenarios[col_idx]
                scenario_data = self.scenarios_data.get(ini_path)
                if scenario_data:
                    panel.set_data(scenario_data.results, scenario_data.params, scenario_data.devise)
        # Forcer un redraw après que le layout soit finalisé
        # (résout le problème de canvas non dimensionné au premier affichage)
        self.after(100, self._force_redraw_all_panels)
        # Synchroniser les échelles Y si tous les scénarios sont du même pays
        self.after(200, self._synchronize_y_scales_if_same_country)

    def _force_redraw_all_panels(self):
        """Force le redraw de tous les panneaux après que le layout soit stable"""
        for panel in self.graph_panels.values():
            if hasattr(panel, 'redraw'):
                panel.redraw()

    def _all_same_country(self) -> bool:
        """Vérifie si tous les scénarios chargés sont du même pays"""
        countries = set()
        for ini_path in self.config.scenarios:
            scenario_data = self.scenarios_data.get(ini_path)
            if scenario_data and hasattr(scenario_data.params, 'nom_pays'):
                countries.add(scenario_data.params.nom_pays)
        return len(countries) == 1

    def _synchronize_y_scales_if_same_country(self):
        """
        Synchronise les échelles Y par LIGNE seulement si tous les scénarios
        sont du même pays. Cela permet de comparer différents scénarios
        d'un même pays sur la même échelle.
        """
        if not self._all_same_country():
            return  # Pays différents : pas de synchronisation

        n_rows = len(self.config.graphs)
        n_cols = len(self.config.scenarios)

        for row_idx in range(n_rows):
            # Collecter les limites Y de tous les graphiques de cette ligne
            y_mins = []
            y_maxs = []

            for col_idx in range(n_cols):
                panel = self.graph_panels.get((row_idx, col_idx))
                if panel and hasattr(panel, 'get_ylim'):
                    try:
                        ylim = panel.get_ylim()
                        if ylim and len(ylim) == 2:
                            y_mins.append(ylim[0])
                            y_maxs.append(ylim[1])
                    except Exception:
                        pass

            # Si on a des limites valides, appliquer l'échelle unifiée
            if y_mins and y_maxs:
                global_ymin = min(y_mins)
                global_ymax = max(y_maxs)

                # Appliquer à tous les graphiques de la ligne
                for col_idx in range(n_cols):
                    panel = self.graph_panels.get((row_idx, col_idx))
                    if panel and hasattr(panel, 'set_ylim'):
                        try:
                            panel.set_ylim(global_ymin, global_ymax)
                        except Exception:
                            pass

    def _add_scenario(self):
        if not self.config.can_add_scenario():
            return
        used = set(self.config.scenarios)
        for ini_path, name in self.available_scenarios:
            if ini_path not in used:
                self.config.scenarios.append(ini_path)
                self._load_scenario(ini_path)
                self._rebuild_grid()
                return
        self._set_status("Tous les scénarios sont déjà utilisés")

    def _remove_scenario_at(self, col_idx: int):
        """Supprime le scénario à l'index donné"""
        if self.config.can_remove_scenario() and col_idx < len(self.config.scenarios):
            self.config.scenarios.pop(col_idx)
            self._rebuild_grid()

    def _add_graph(self):
        if not self.config.can_add_graph():
            return
        used = set(self.config.graphs)
        for graph_id in self.graph_files.keys():
            if graph_id not in used:
                self.config.graphs.append(graph_id)
                self._rebuild_grid()
                return
        if self.graph_files:
            self.config.graphs.append(list(self.graph_files.keys())[0])
            self._rebuild_grid()

    def _remove_graph_at(self, row_idx: int):
        """Supprime le graphique à l'index donné"""
        if self.config.can_remove_graph() and row_idx < len(self.config.graphs):
            self.config.graphs.pop(row_idx)
            self._rebuild_grid()

    def _update_status(self):
        n = len(self.config.scenarios)
        g = len(self.config.graphs)
        self._set_status(f"Comparaison: {n} pays × {g} graphiques")

    def update_graph_files(self, graph_files: Dict[str, Any]):
        self.graph_files = graph_files
        for panel in self.graph_panels.values():
            panel.update_graph_files(graph_files)

    def get_config(self) -> ComparisonConfig:
        return self.config
