# Spécification : Mode Comparaison de Scénarios

**Version** : 1.0
**Date** : 2024-12-15
**Statut** : Validé pour implémentation

---

## 1. Vue d'ensemble

### 1.1 Objectif

Permettre la comparaison visuelle de plusieurs scénarios de simulation (pays, hypothèses, variantes) via une grille de graphiques juxtaposés, sans superposition de courbes.

### 1.2 Principe fondamental

- **Colonnes** = Scénarios (ex: France, Belgique, Allemagne)
- **Lignes** = Métriques (ex: Dette % PIB, Différentiel, PIB indexé)
- **Juxtaposition**, pas superposition

### 1.3 Contraintes non négociables

| Règle | Description |
|-------|-------------|
| Pas de superposition | Chaque cellule = 1 graphique = 1 scénario × 1 métrique |
| Pas d'édition en comparaison | Scénarios figés (lecture seule) |
| Pas de recalcul implicite | Résultats pré-calculés uniquement |
| Axe Y automatique | Selon le type de métrique (normalisée vs absolue) |
| Honnêteté économique | Interdiction de forcer un axe Y commun sur des absolus hétérogènes |

---

## 2. Architecture des modes

### 2.1 Deux modes distincts

```
┌─────────────────────────────────────────────────────────────┐
│                      APPLICATION                            │
├─────────────────────────┬───────────────────────────────────┤
│     MODE NORMAL         │       MODE COMPARAISON            │
├─────────────────────────┼───────────────────────────────────┤
│ • 1 scénario actif      │ • N scénarios (2-4 colonnes)      │
│ • Paramètres éditables  │ • Paramètres figés (lecture)      │
│ • Grille N×M flexible   │ • Grille N×M flexible             │
│ • Graphiques libres     │ • Même métrique par ligne         │
│ • Recalcul auto (Live)  │ • Pas de recalcul                 │
└─────────────────────────┴───────────────────────────────────┘
```

### 2.2 Persistance des configurations

Chaque mode sauvegarde sa propre configuration de grille :

```python
# État sauvegardé séparément
normal_mode_config = {
    'rows': 2,
    'cols': 2,
    'cells': [
        {'graph': 'evolution_pib'},
        {'graph': 'evolution_dettes_pct'},
        {'graph': 'flux_pensions'},
        {'graph': 'comparaison_dettes'}
    ]
}

comparison_mode_config = {
    'rows': 2,
    'cols': 3,
    'scenarios': ['france_base', 'belgique_base', 'allemagne_base'],
    'metrics': ['DPPC', 'DIFF']  # Codes 4 lettres
}
```

### 2.3 Basculement entre modes

- Le passage d'un mode à l'autre **restaure** la configuration précédente de ce mode
- Aucune perte de configuration

---

## 3. Interface utilisateur

### 3.1 Layout de la grille (mode comparaison)

```
┌────┬───────────────┬───────────────┬───────────────┬───┐
│    │  France ▼     │  Belgique ▼   │  Allemagne ▼  │[+]│
├────┼───────────────┼───────────────┼───────────────┼───┤
│ ▼  │  [graphique]  │  [graphique]  │  [graphique]  │   │
├────┼───────────────┼───────────────┼───────────────┼───┤
│ ▼  │  [graphique]  │  [graphique]  │  [graphique]  │   │
├────┼───────────────┼───────────────┼───────────────┼───┤
│[+] │               │               │               │   │
└────┴───────────────┴───────────────┴───────────────┴───┘
```

### 3.2 Composants de l'interface

#### 3.2.1 Sélecteur de scénario (en-tête de colonne)

```
┌─────────────────────────────┐
│  France ▼              [×]  │  ← [×] visible au survol
└─────────────────────────────┘
        │
        ▼ Clic sur ▼
┌─────────────────────────────┐
│ ✓ france_base               │
│   france_optimiste          │
│   france_pessimiste         │
│ ─────────────────────────── │
│   belgique_base             │
│   belgique_optimiste        │
│   ...                       │
└─────────────────────────────┘
```

#### 3.2.2 Sélecteur de métrique (en-tête de ligne)

```
┌────┐
│ ▼  │  ← Icône compacte (~30px)
└────┘
  │
  │ Survol (après 500ms) → Tooltip: "Dette publique (% PIB)"
  │
  ▼ Clic
┌─────────────────────────────┐
│ MÉTRIQUES NORMALISÉES       │
│ (axe Y commun)              │
│ ─────────────────────────── │
│ ✓ Dette publique % PIB      │
│   Dette transition % PIB    │
│   Dette totale % PIB        │
│   Dette implicite % PIB     │
│   Différentiel % PIB        │
│   Surplus budgétaire %      │
│ ─────────────────────────── │
│ MÉTRIQUES INDEXÉES          │
│ (axe Y commun, base 100)    │
│ ─────────────────────────── │
│   PIB indexé                │
│   Dette indexée             │
│   Pouvoir d'achat indexé    │
│ ─────────────────────────── │
│ 🗑️ Supprimer cette ligne    │
└─────────────────────────────┘
```

#### 3.2.3 Boutons [+] pour ajouter

- **[+] en fin de ligne d'en-têtes** : Ajoute une colonne (nouveau scénario)
- **[+] en bas de la colonne de gauche** : Ajoute une ligne (nouvelle métrique)

#### 3.2.4 Bouton [×] pour supprimer

- Apparaît au **survol** de l'en-tête de colonne ou ligne
- Clic → Supprime immédiatement (pas de confirmation)

### 3.3 Harmonisation avec le mode normal

Le mode normal adopte le même pattern [+]/[×] :

```
┌───────────────┬───────────────┬───┐
│  [graphique]  │  [graphique]  │[+]│
├───────────────┼───────────────┼───┤
│  [graphique]  │  [graphique]  │   │
├───────────────┼───────────────┼───┤
│      [+]      │      [+]      │   │
└───────────────┴───────────────┴───┘
```

- Chaque cellule garde son sélecteur de graphique individuel
- Les menus existants pour ajouter des graphiques sont supprimés

---

## 4. Gestion des axes

### 4.1 Axe X (temps)

- **Toujours commun** à tous les graphiques de la grille
- Plage = union des plages de tous les scénarios
- Si un scénario est plus court, ses courbes s'arrêtent (pas d'extrapolation)

### 4.2 Axe Y — Règle automatique

#### 4.2.1 Métriques normalisées → Axe Y COMMUN

| Code | Métrique | Unité |
|------|----------|-------|
| DPPC | Dette publique | % PIB |
| DTPC | Dette transition | % PIB |
| TTPC | Dette totale | % PIB |
| DIPC | Dette implicite | % PIB |
| DIFF | Différentiel | % PIB |
| SBPC | Surplus budgétaire | % PIB |

**Comportement** : Tous les graphiques de la ligne partagent la même échelle Y.

#### 4.2.2 Métriques indexées (à créer) → Axe Y COMMUN

| Code | Métrique | Description |
|------|----------|-------------|
| PIBI | PIB indexé | Base 100 à l'année 0 |
| DETI | Dette indexée | Base 100 à l'année 0 |
| PACI | Pouvoir d'achat indexé | Base 100 à l'année 0 |

**Comportement** : Échelle Y commune car comparable (base 100).

#### 4.2.3 Métriques absolues → Axe Y INDÉPENDANT

| Code | Métrique | Unité |
|------|----------|-------|
| PIBS | PIB | Mds devise |
| DPUB | Dette publique | Mds devise |
| DIMP | Dette implicite | Mds devise |
| FLUX | Flux pensions | Mds devise |
| INTR | Intérêts | Mds devise |

**Comportement** :
- Chaque colonne a sa propre échelle Y
- **Avertissement visuel** : label "(échelle propre)" sur l'axe

### 4.3 Indicateur visuel du type d'axe

Dans le dropdown de sélection de métrique :
```
│ MÉTRIQUES NORMALISÉES       │  ← Section avec icône 📊
│ (axe Y commun)              │
│ ─────────────────────────── │
│   Dette publique % PIB      │
```

Dans le tooltip :
```
"Dette publique (% PIB) — axe commun"
"PIB (Mds €) — échelle propre par pays"
```

---

## 5. Métriques dérivées

### 5.1 Principe

Si une comparaison n'est pas économiquement valide, créer une métrique dérivée plutôt que forcer une visualisation trompeuse.

### 5.2 Métriques à créer

#### 5.2.1 Indexation base 100

```python
def calculer_metrique_indexee(resultats: List[Dict], champ: str) -> List[Dict]:
    """
    Transforme une métrique absolue en métrique indexée base 100.

    Args:
        resultats: Liste des résultats année par année
        champ: Nom du champ à indexer (ex: 'pib', 'dette_publique')

    Returns:
        Liste avec nouveau champ '{champ}_indexe' ajouté
    """
    valeur_base = resultats[0][champ]  # Année 0 = base 100

    for r in resultats:
        r[f'{champ}_indexe'] = (r[champ] / valeur_base) * 100 if valeur_base else 0

    return resultats
```

#### 5.2.2 Nouvelles métriques

| Nouveau code | Formule | Type |
|--------------|---------|------|
| PIBI | `pib / pib[0] × 100` | Indexé (axe commun) |
| DPBI | `dette_publique / dette_publique[0] × 100` | Indexé (axe commun) |
| DTBI | `dette_transition / max(dette_transition) × 100` | Indexé (axe commun) |
| DIBI | `dette_implicite / dette_implicite[0] × 100` | Indexé (axe commun) |
| PACI | `pouvoir_achat / pouvoir_achat[0] × 100` | Indexé (axe commun) |

### 5.3 Fichiers YAML des nouvelles métriques

Créer les fichiers dans `gui/graphiques/` :

**pib_indexe.yaml**
```yaml
title: "PIB indexé (base 100)"
type: line
x_axis:
  field: ANNE
  label: "Année"
y_axis:
  label: "Indice (base 100 = année 0)"
  min: 0
curves:
  - field: PIBI
    label: "PIB"
    color: "#4A90E2"
```

---

## 6. Structures de données

### 6.1 Configuration du mode comparaison

```python
@dataclass
class ComparisonConfig:
    """Configuration du mode comparaison"""
    scenarios: List[str]           # Chemins relatifs des fichiers .ini
    metrics: List[str]             # Codes 4 lettres des métriques

    # Contraintes
    MIN_COLS: int = 2
    MAX_COLS: int = 4
    MIN_ROWS: int = 1
    MAX_ROWS: int = 4


@dataclass
class ComparisonState:
    """État complet du mode comparaison"""
    config: ComparisonConfig
    results: Dict[str, List[Dict]]  # scenario_name -> resultats
    params: Dict[str, Parametres]   # scenario_name -> parametres
```

### 6.2 Registre des métriques

```python
@dataclass
class MetricDefinition:
    """Définition d'une métrique"""
    code: str           # Code 4 lettres (ex: 'DPPC')
    field: str          # Nom du champ dans resultats (ex: 'dette_publique_pct')
    label: str          # Nom affiché (ex: 'Dette publique (% PIB)')
    category: str       # 'normalized', 'indexed', 'absolute'
    unit: str           # '%', 'Mds', 'index'
    derived_from: str   # Si calculé: champ source. Sinon: None


METRICS_REGISTRY = {
    # Normalisées (existantes)
    'DPPC': MetricDefinition('DPPC', 'dette_publique_pct', 'Dette publique', 'normalized', '%', None),
    'DTPC': MetricDefinition('DTPC', 'dette_transition_pct', 'Dette transition', 'normalized', '%', None),
    'TTPC': MetricDefinition('TTPC', 'dette_publique_totale_pct', 'Dette totale', 'normalized', '%', None),
    'DIPC': MetricDefinition('DIPC', 'dette_implicite_pct', 'Dette implicite', 'normalized', '%', None),
    'DIFF': MetricDefinition('DIFF', 'differentiel_pct', 'Différentiel', 'normalized', '%', None),

    # Indexées (à créer)
    'PIBI': MetricDefinition('PIBI', 'pib_indexe', 'PIB indexé', 'indexed', 'index', 'pib'),
    'DPBI': MetricDefinition('DPBI', 'dette_publique_indexee', 'Dette pub. indexée', 'indexed', 'index', 'dette_publique'),

    # Absolues (existantes)
    'PIBS': MetricDefinition('PIBS', 'pib', 'PIB', 'absolute', 'Mds', None),
    'DPUB': MetricDefinition('DPUB', 'dette_publique', 'Dette publique', 'absolute', 'Mds', None),
    # ...
}
```

---

## 7. Comportements détaillés

### 7.1 Chargement d'un scénario en comparaison

```
1. Utilisateur sélectionne un scénario dans le dropdown
2. Vérifier si déjà calculé (cache)
3. Si non calculé :
   a. Charger les paramètres depuis le .ini
   b. Exécuter la simulation
   c. Calculer les métriques dérivées (indexées)
   d. Stocker dans le cache
4. Rafraîchir tous les graphiques de la colonne
```

### 7.2 Changement de métrique d'une ligne

```
1. Utilisateur clique sur ▼ d'une ligne
2. Afficher le dropdown avec les métriques groupées
3. Utilisateur sélectionne une métrique
4. Déterminer le type d'axe Y (commun ou indépendant)
5. Rafraîchir tous les graphiques de la ligne
6. Si axe commun : recalculer l'échelle Y globale
```

### 7.3 Tooltip avec délai

```python
def show_tooltip(widget, text, delay_ms=500):
    """Affiche un tooltip après un délai"""
    tooltip_id = None

    def schedule_tooltip(event):
        nonlocal tooltip_id
        tooltip_id = widget.after(delay_ms, lambda: display_tooltip(event.x_root, event.y_root, text))

    def cancel_tooltip(event):
        nonlocal tooltip_id
        if tooltip_id:
            widget.after_cancel(tooltip_id)
            tooltip_id = None
        hide_tooltip()

    widget.bind('<Enter>', schedule_tooltip)
    widget.bind('<Leave>', cancel_tooltip)
    widget.bind('<Button-1>', cancel_tooltip)  # Annuler si clic
```

---

## 8. Plan d'implémentation

### Phase 1 : Infrastructure (priorité haute)

1. [ ] Créer `ComparisonConfig` et `ComparisonState`
2. [ ] Créer `METRICS_REGISTRY` avec toutes les métriques
3. [ ] Implémenter le calcul des métriques indexées
4. [ ] Créer les fichiers YAML des nouvelles métriques

### Phase 2 : UI Mode Comparaison (priorité haute)

5. [ ] Créer `ComparisonGridPanel` (nouvelle classe)
6. [ ] Implémenter les sélecteurs de scénario (en-tête colonne)
7. [ ] Implémenter les sélecteurs de métrique (en-tête ligne avec ▼)
8. [ ] Implémenter le système de tooltip avec délai
9. [ ] Implémenter les boutons [+] et [×]

### Phase 3 : Harmonisation Mode Normal (priorité moyenne)

10. [ ] Remplacer les menus d'ajout de graphique par [+]
11. [ ] Ajouter les boutons [×] au survol
12. [ ] Persister les configurations séparément

### Phase 4 : Gestion des axes (priorité haute)

13. [ ] Implémenter la détection automatique du type d'axe
14. [ ] Implémenter le calcul de l'échelle Y commune par ligne
15. [ ] Ajouter les indicateurs visuels (échelle propre)

### Phase 5 : Finitions (priorité basse)

16. [ ] Menu de basculement entre modes
17. [ ] Sauvegarde/restauration des configurations
18. [ ] Tests et validation

---

## 9. Fichiers à créer/modifier

### Nouveaux fichiers

| Fichier | Description |
|---------|-------------|
| `gui/comparison_mode.py` | Classes ComparisonConfig, ComparisonState, ComparisonGridPanel |
| `gui/metrics_registry.py` | Registre des métriques et fonctions de calcul |
| `gui/graphiques/pib_indexe.yaml` | Définition graphique PIB indexé |
| `gui/graphiques/dette_indexee.yaml` | Définition graphique dette indexée |
| `gui/graphiques/pouvoir_achat_indexe.yaml` | Définition graphique pouvoir d'achat indexé |

### Fichiers à modifier

| Fichier | Modifications |
|---------|---------------|
| `gui/simulateur_gui.py` | Ajouter basculement de mode, harmoniser [+]/[×] |
| `code/transition_pensions.py` | Ajouter calcul métriques indexées dans `simuler()` |

---

## 10. Contraintes techniques

### 10.1 Performance

- Cache des résultats de simulation par scénario
- Ne pas recalculer si paramètres inchangés
- Rafraîchissement graphique optimisé (batch)

### 10.2 Mémoire

- Maximum 16 graphiques (4×4) × 4 scénarios = ~64 courbes
- Résultats : ~150 années × 20 champs × 4 scénarios = ~12 000 valeurs
- Impact mémoire négligeable

### 10.3 Compatibilité

- Python 3.8+
- tkinter (inclus dans Python)
- matplotlib 3.x
- Pas de dépendances supplémentaires

---

## Annexe A : Codes des métriques

### Existants

| Code | Champ | Description |
|------|-------|-------------|
| ANNE | annee | Année de simulation |
| PIBS | pib | PIB (Mds) |
| DIFF | differentiel_pct | Différentiel (% PIB) |
| INTR | interets_totaux | Intérêts (Mds/an) |
| DPUB | dette_publique | Dette publique (Mds) |
| DTRA | dette_transition | Dette transition (Mds) |
| DTOT | dette_publique_totale | Dette totale (Mds) |
| DPPC | dette_publique_pct | Dette publique (% PIB) |
| DTPC | dette_transition_pct | Dette transition (% PIB) |
| TTPC | dette_publique_totale_pct | Dette totale (% PIB) |
| DIMP | dette_implicite | Dette implicite (Mds) |
| DIPC | dette_implicite_pct | Dette implicite (% PIB) |
| FLUX | flux_pensions | Flux pensions (Mds/an) |
| NRET | nombre_retraites | Nombre de retraités |

### À créer

| Code | Champ | Description |
|------|-------|-------------|
| PIBI | pib_indexe | PIB indexé (base 100) |
| DPBI | dette_publique_indexee | Dette pub. indexée (base 100) |
| DTBI | dette_transition_indexee | Dette trans. indexée (base 100) |
| DIBI | dette_implicite_indexee | Dette impl. indexée (base 100) |
| PACI | pouvoir_achat_indexe | Pouvoir d'achat indexé (base 100) |

---

## Annexe B : Maquette complète

```
┌─────────────────────────────────────────────────────────────────────────────┐
│ Simulateur Transition Pensions                    [Mode: Comparaison ▼]    │
├─────────────────────────────────────────────────────────────────────────────┤
│                                                                             │
│  ┌────┬─────────────────┬─────────────────┬─────────────────┬───┐          │
│  │    │   France ▼  [×] │  Belgique ▼ [×] │ Allemagne ▼ [×] │[+]│          │
│  ├────┼─────────────────┼─────────────────┼─────────────────┼───┤          │
│  │ ▼  │                 │                 │                 │   │          │
│  │    │   ┌─────────┐   │   ┌─────────┐   │   ┌─────────┐   │   │          │
│  │    │   │ Graph 1 │   │   │ Graph 2 │   │   │ Graph 3 │   │   │          │
│  │    │   │ DPPC    │   │   │ DPPC    │   │   │ DPPC    │   │   │          │
│  │    │   └─────────┘   │   └─────────┘   │   └─────────┘   │   │          │
│  ├────┼─────────────────┼─────────────────┼─────────────────┼───┤          │
│  │ ▼  │                 │                 │                 │   │          │
│  │    │   ┌─────────┐   │   ┌─────────┐   │   ┌─────────┐   │   │          │
│  │    │   │ Graph 4 │   │   │ Graph 5 │   │   │ Graph 6 │   │   │          │
│  │    │   │ DIFF    │   │   │ DIFF    │   │   │ DIFF    │   │   │          │
│  │    │   └─────────┘   │   └─────────┘   │   └─────────┘   │   │          │
│  ├────┼─────────────────┼─────────────────┼─────────────────┼───┤          │
│  │[+] │                 │                 │                 │   │          │
│  └────┴─────────────────┴─────────────────┴─────────────────┴───┘          │
│                                                                             │
├─────────────────────────────────────────────────────────────────────────────┤
│ Status: 3 scénarios chargés, 2 métriques affichées                         │
└─────────────────────────────────────────────────────────────────────────────┘
```
