#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Simulation de la transition d'un système de pensions par répartition vers capitalisation

Ce script simule année par année la transition complète et génère:
- Un fichier CSV avec toutes les données
- Un fichier Markdown avec tableau synthétique
- Des graphiques de visualisation (optionnel)
"""

import math
import csv
import configparser
import sys
import xml.etree.ElementTree as ET
from dataclasses import dataclass, field
from typing import List, Tuple, Dict, Optional


@dataclass
class Parametres:
    """
    Classe contenant tous les paramètres de la simulation
    Facilement modifiable pour simuler différents pays
    """
    
    # === PARAMÈTRES ÉCONOMIQUES GÉNÉRAUX ===
    pib_initial: float = 850.0  # Milliards €
    dette_publique_initiale: float = 884.0  # Milliards €
    nombre_travailleurs_actifs: int = 5_000_000  # Nombre de travailleurs pour calcul impact abattement

    # Taux de croissance du PIB
    taux_croissance_base: float = 0.015  # 1.5%/an
    bonus_croissance_1_10: float = 0.02  # +2% années 1-10
    bonus_croissance_11_20: float = 0.01  # +1% années 11-20
    bonus_croissance_20_plus: float = 0.005  # +0.5% années 20+
    
    # === PARAMÈTRES SYSTÈME DE PENSIONS ===
    dette_implicite_initiale: float = 2100.0  # Milliards € (avant réduction)
    reduction_solidaire: float = 0.10  # 10% de réduction progressive
    
    nombre_retraites_initiaux: int = 2_500_000
    pension_moyenne_annuelle: float = 15_000.0  # €
    nouveaux_retraites_par_an: int = 100_000
    
    duree_carriere: int = 40  # années
    age_depart_retraite: int = 65  # ans
    age_esperance_vie: int = 85  # ans
    
    # === PARAMÈTRES MORTALITÉ ===
    taux_mortalite_initial: float = 0.05  # 5%/an
    increment_mortalite_annuel: float = 0.003  # +0.3%/an
    
    # === PARAMÈTRES DROITS RÉPARTITION ===
    methode_droits: str = 'quadratique'  # 'quadratique', 'exponentielle', 'paliers'
    parametre_methode_droits: float = 1.5  # Exposant pour quadratique
    # Pour paliers: [(5, 0.20), (15, 0.50), (25, 1.00), (40, 1.60)]
    
    # === PARAMÈTRES FINANCEMENT ===
    privatisations_totales: float = 195.0  # Milliards €
    
    # Différentiel avec décroissance progressive
    differentiel_initial: float = 0.10  # 10% du PIB au départ
    duree_decroissance_differentiel: int = 40  # Décroissance sur 40 ans
    methode_differentiel: str = 'lineaire'  # 'lineaire', 'quadratique_inverse', 'exponentielle_inverse', 'paliers'
    parametre_methode_differentiel: float = -1  # -1 = valeur par défaut selon méthode
    
    remboursement_dette_pub_pct_pib: float = 0.01  # 1% du PIB/an
    
    # === SURPLUS BUDGÉTAIRE MINIMAL CONSTITUTIONNEL ===
    # Ce surplus prend le relais du différentiel pour rembourser la dette de transition
    # lorsque le différentiel atteint 0 ou est insuffisant
    surplus_budgetaire_minimal_pct_pib: float = 0.02  # 2% du PIB minimal
    surplus_max_pour_dette_transition_pct: float = 1.0  # 100% du surplus peut aller à la dette de transition

    # === PARAMÈTRES TAUX D'INTÉRÊT ===
    # Format: [(ratio_max, taux_min, taux_max), ...]
    seuils_taux_interet: List[Tuple[float, float, float]] = field(default_factory=lambda: [
        (0.60, 0.015, 0.020),   # ratio < 60%: taux 1.5-2%
        (0.90, 0.020, 0.025),   # ratio 60-90%: taux 2-2.5%
        (1.20, 0.025, 0.030),   # ratio 90-120%: taux 2.5-3%
        (9.99, 0.030, 0.040),   # ratio > 120%: taux 3-4%
    ])
    
    # === CONFIGURATION PAYS ===
    nom_pays: str = "Belgique"
    
    # === AFFICHAGE ===
    decimales_affichage: int = 2  # Nombre de décimales dans le tableau Markdown
    
    # === ANALYSE IMPACT SALAIRES ===
    generer_tableau_salaires: bool = True
    fichier_tableau_salaires: str = "transition_pensions_salaires.md"
    
    salaires_analyses: List[float] = field(default_factory=lambda: [2000, 3000, 4000, 5000, 7000, 10000])  # €/mois
    taux_flat_tax: float = 0.25  # 25%
    abattement_forfaitaire: float = 0.0  # €/mois - les premiers euros non taxés
    
    # Assurances mensuelles (4 postes séparés)
    assurance_sante: float = 80.0      # €/mois
    assurance_chomage: float = 40.0    # €/mois
    assurance_pension: float = 60.0    # €/mois (capitalisation)
    assurance_education: float = 50.0  # €/mois (modulable selon nb enfants)
    
    # Système actuel (année -1) pour comparaison
    prelevement_actuel_total: float = 0.45  # 45% (cotisations + impôts actuels)

    # Taxes indirectes par tranche de salaire (% du net actuel)
    # Ces taxes régressives (TVA, accises, etc.) sont abolies avec la flat tax
    taxes_indirectes: Dict[float, float] = field(default_factory=lambda: {
        2000: 0.16,   # 16% pour salaire brut 2000€
        3000: 0.14,   # 14% pour salaire brut 3000€
        4000: 0.12,   # 12% pour salaire brut 4000€
        5000: 0.11,   # 11% pour salaire brut 5000€
        7000: 0.09,   # 9% pour salaire brut 7000€
        10000: 0.08,  # 8% pour salaire brut 10000€
    })


def calculer_taux_mortalite(annee_depuis_retraite: int, 
                            taux_initial: float, 
                            increment_annuel: float) -> float:
    """
    Calcule le taux de mortalité progressif
    
    Parameters:
    -----------
    annee_depuis_retraite : int
        Nombre d'années depuis le départ en retraite (0, 1, 2, ...)
    taux_initial : float
        Taux de mortalité la première année (ex: 0.05 = 5%)
    increment_annuel : float
        Augmentation du taux chaque année (ex: 0.003 = +0.3%)
    
    Returns:
    --------
    float : Taux de mortalité pour cette année
    """
    return taux_initial + (annee_depuis_retraite * increment_annuel)


def calculer_droits_repartition(annee_transition: int, 
                                duree_carriere: int = 40,
                                methode: str = 'quadratique', 
                                parametre_methode = -1) -> float:
    """
    Calcule les droits à répartition des nouveaux retraités
    selon une courbe d'amortissement non-linéaire
    
    Parameters:
    -----------
    annee_transition : int
        Année depuis le début de la transition (1, 2, 3, ...)
    duree_carriere : int
        Durée totale de carrière (défaut: 40 ans)
    methode : str
        'quadratique' : décroissance par puissance
        'exponentielle' : décroissance exponentielle
        'paliers' : par paliers progressifs
    parametre_methode : float ou list
        -1 (défaut) : utilise valeurs par défaut selon méthode
        Pour 'quadratique' : exposant (défaut 1.5 si -1)
        Pour 'exponentielle' : taux de décroissance (défaut 0.05 si -1)
        Pour 'paliers' : liste de tuples [(annee_fin, taux_perte_annuel), ...]
    
    Returns:
    --------
    float : droits en nombre d'années sur duree_carriere
    """
    if annee_transition > duree_carriere:
        return 0.0
    
    if methode == 'quadratique':
        exposant = 1.5 if parametre_methode == -1 else parametre_methode
        annees_restantes = duree_carriere - (annee_transition - 1)
        ratio = (annees_restantes / duree_carriere) ** exposant
        droits = duree_carriere * ratio
        
    elif methode == 'exponentielle':
        taux = 0.05 if parametre_methode == -1 else parametre_methode
        droits = duree_carriere * math.exp(-taux * (annee_transition - 1))
        if annee_transition == duree_carriere:
            droits = 0.0
            
    elif methode == 'paliers':
        # Défaut: paliers progressifs
        if parametre_methode == -1:
            paliers = [(5, 0.20), (15, 0.50), (25, 1.00), (40, 1.60)]
        else:
            paliers = parametre_methode
        
        # Calcul de la perte cumulée selon les paliers
        perte_cumulee = 0.0
        annee_debut_palier = 1
        
        for annee_fin_palier, taux_perte in paliers:
            if annee_transition <= annee_fin_palier:
                # On est dans ce palier
                annees_dans_palier = annee_transition - annee_debut_palier
                perte_cumulee += annees_dans_palier * taux_perte
                break
            else:
                # On a passé ce palier complètement
                annees_dans_palier = annee_fin_palier - annee_debut_palier + 1
                perte_cumulee += annees_dans_palier * taux_perte
                annee_debut_palier = annee_fin_palier + 1
        
        droits = duree_carriere - perte_cumulee
        
    else:
        raise ValueError(f"Méthode '{methode}' non reconnue")
    
    return max(0.0, droits)


def calculer_plafond_differentiel(annee: int,
                                   differentiel_initial: float = 0.10,
                                   duree_decroissance: int = 40,
                                   methode: str = 'lineaire',
                                   parametre_methode = -1) -> float:
    """
    Calcule le plafond du différentiel avec décroissance progressive
    
    Parameters:
    -----------
    annee : int
        Année depuis le début (1, 2, 3, ...)
    differentiel_initial : float
        Différentiel de départ (ex: 0.10 = 10%)
    duree_decroissance : int
        Durée sur laquelle le différentiel décroît jusqu'à 0
    methode : str
        'quadratique_inverse' : décroissance lente début, rapide fin
        'lineaire' : décroissance constante
        'exponentielle_inverse' : décroissance très lente début
        'paliers' : par paliers
    parametre_methode : float ou list
        Pour 'quadratique_inverse' : exposant (défaut 1.5)
        Pour 'exponentielle_inverse' : taux (défaut 0.05)
        Pour 'paliers' : liste [(annee, differentiel), ...]
    
    Returns:
    --------
    float : Plafond du différentiel (en fraction, ex: 0.10 = 10%)
    """
    if annee > duree_decroissance:
        return 0.0
    
    if methode == 'quadratique_inverse':
        # Décroissance douce au début, rapide à la fin
        exposant = 1.5 if parametre_methode == -1 else parametre_methode
        ratio_restant = (duree_decroissance - annee) / duree_decroissance
        plafond = differentiel_initial * (ratio_restant ** exposant)
        
    elif methode == 'lineaire':
        # Décroissance constante
        plafond = differentiel_initial * (1 - annee / duree_decroissance)
        
    elif methode == 'exponentielle_inverse':
        # Décroissance très lente au début
        taux = 0.05 if parametre_methode == -1 else parametre_methode
        plafond = differentiel_initial * math.exp(-taux * annee)
        
    elif methode == 'paliers':
        # Par paliers
        if parametre_methode == -1:
            # Valeurs par défaut : décroissance par paliers de 5 ans
            paliers = [
                (5, 0.095),
                (10, 0.090),
                (15, 0.080),
                (20, 0.065),
                (25, 0.050),
                (30, 0.035),
                (35, 0.020),
                (40, 0.000)
            ]
        else:
            paliers = parametre_methode
        
        # Trouver le palier correspondant
        plafond = differentiel_initial
        for annee_palier, diff_palier in paliers:
            if annee <= annee_palier:
                plafond = diff_palier
                break
    
    else:
        raise ValueError(f"Méthode '{methode}' non reconnue")
    
    return max(0.0, plafond)


def calculer_taux_interet(ratio_dette_pib: float, 
                          seuils: List[Tuple[float, float, float]]) -> float:
    """
    Calcule le taux d'intérêt en fonction du ratio dette/PIB
    
    Parameters:
    -----------
    ratio_dette_pib : float
        Ratio dette totale / PIB
    seuils : List[Tuple[float, float, float]]
        Liste de (ratio_max, taux_min, taux_max)
    
    Returns:
    --------
    float : Taux d'intérêt annuel
    """
    for ratio_max, taux_min, taux_max in seuils:
        if ratio_dette_pib <= ratio_max:
            # Interpolation linéaire dans le palier
            if len(seuils) > 1:
                idx = seuils.index((ratio_max, taux_min, taux_max))
                if idx > 0:
                    ratio_min_palier = seuils[idx-1][0]
                else:
                    ratio_min_palier = 0.0
                
                ratio_dans_palier = (ratio_dette_pib - ratio_min_palier) / (ratio_max - ratio_min_palier)
                return taux_min + ratio_dans_palier * (taux_max - taux_min)
            return taux_min
    
    # Si au-delà de tous les seuils, retourner le taux max du dernier palier
    return seuils[-1][2]


def charger_configuration(fichier_ini: str) -> 'Parametres':
    """
    Charge la configuration depuis un fichier INI
    
    Parameters:
    -----------
    fichier_ini : str
        Chemin vers le fichier .ini
    
    Returns:
    --------
    Parametres : Objet Parametres avec les valeurs chargées
    """
    config = configparser.ConfigParser()
    config.read(fichier_ini, encoding='utf-8')
    
    def get_float(section, key, default):
        try:
            return config.getfloat(section, key)
        except:
            return default
    
    def get_int(section, key, default):
        try:
            return config.getint(section, key)
        except:
            return default
    
    def get_str(section, key, default):
        try:
            return config.get(section, key)
        except:
            return default
    
    def get_bool(section, key, default):
        try:
            return config.getboolean(section, key)
        except:
            return default
    
    def get_list_float(section, key, default):
        try:
            valeur = config.get(section, key)
            return [float(x.strip()) for x in valeur.split(',')]
        except:
            return default
    
    def get_seuils_taux(config):
        """Parse les seuils de taux d'intérêt"""
        seuils = []
        section = 'TauxInteret'
        for key in config.options(section):
            if key.startswith('seuil_'):
                valeur = config.get(section, key)
                parts = [float(x.strip()) for x in valeur.split(',')]
                if len(parts) == 3:
                    seuils.append(tuple(parts))
        return seuils if seuils else [
            (0.60, 0.015, 0.020),
            (0.90, 0.020, 0.025),
            (1.20, 0.025, 0.030),
            (9.99, 0.030, 0.040)
        ]
    
    # Charger les paramètres
    params = Parametres(
        # Pays
        nom_pays=get_str('Pays', 'nom', 'Belgique'),
        
        # Affichage
        decimales_affichage=get_int('Affichage', 'decimales_affichage', 2),
        
        # Économie
        pib_initial=get_float('Economie', 'pib_initial', 850.0),
        dette_publique_initiale=get_float('Economie', 'dette_publique_initiale', 884.0),
        nombre_travailleurs_actifs=get_int('Economie', 'nombre_travailleurs_actifs', 5000000),
        taux_croissance_base=get_float('Economie', 'taux_croissance_base', 0.015),
        bonus_croissance_1_10=get_float('Economie', 'bonus_croissance_1_10', 0.02),
        bonus_croissance_11_20=get_float('Economie', 'bonus_croissance_11_20', 0.01),
        bonus_croissance_20_plus=get_float('Economie', 'bonus_croissance_20_plus', 0.005),
        
        # Pensions
        dette_implicite_initiale=get_float('Pensions', 'dette_implicite_initiale', 2100.0),
        reduction_solidaire=get_float('Pensions', 'reduction_solidaire', 0.10),
        nombre_retraites_initiaux=get_int('Pensions', 'nombre_retraites_initiaux', 2500000),
        pension_moyenne_annuelle=get_float('Pensions', 'pension_moyenne_annuelle', 15000.0),
        nouveaux_retraites_par_an=get_int('Pensions', 'nouveaux_retraites_par_an', 100000),
        duree_carriere=get_int('Pensions', 'duree_carriere', 40),
        age_depart_retraite=get_int('Pensions', 'age_depart_retraite', 65),
        age_esperance_vie=get_int('Pensions', 'age_esperance_vie', 85),
        
        # Mortalité
        taux_mortalite_initial=get_float('Mortalite', 'taux_mortalite_initial', 0.05),
        increment_mortalite_annuel=get_float('Mortalite', 'increment_mortalite_annuel', 0.003),
        
        # Droits répartition
        methode_droits=get_str('DroitsRepartition', 'methode_droits', 'quadratique'),
        parametre_methode_droits=get_float('DroitsRepartition', 'parametre_methode_droits', 1.5),
        
        # Financement
        privatisations_totales=get_float('Financement', 'privatisations_totales', 195.0),
        differentiel_initial=get_float('Financement', 'differentiel_initial', 0.10),
        duree_decroissance_differentiel=get_int('Financement', 'duree_decroissance_differentiel', 40),
        methode_differentiel=get_str('Financement', 'methode_differentiel', 'lineaire'),
        parametre_methode_differentiel=get_float('Financement', 'parametre_methode_differentiel', -1),
        remboursement_dette_pub_pct_pib=get_float('Financement', 'remboursement_dette_pub_pct_pib', 0.01),
        surplus_budgetaire_minimal_pct_pib=get_float('Financement', 'surplus_budgetaire_minimal_pct_pib', 0.02),
        surplus_max_pour_dette_transition_pct=get_float('Financement', 'surplus_max_pour_dette_transition_pct', 1.0),
        
        # Taux d'intérêt
        seuils_taux_interet=get_seuils_taux(config),
        
        # Salaires
        generer_tableau_salaires=get_bool('Salaires', 'generer_tableau_salaires', True),
        fichier_tableau_salaires=get_str('Salaires', 'fichier_tableau_salaires', 'transition_pensions_salaires.md'),
        salaires_analyses=get_list_float('Salaires', 'salaires_analyses', [2000, 3000, 4000, 5000, 7000, 10000]),
        taux_flat_tax=get_float('Salaires', 'taux_flat_tax', 0.25),
        abattement_forfaitaire=get_float('Salaires', 'abattement_forfaitaire', 0.0),
        assurance_sante=get_float('Salaires', 'assurance_sante', 80.0),
        assurance_chomage=get_float('Salaires', 'assurance_chomage', 40.0),
        assurance_pension=get_float('Salaires', 'assurance_pension', 60.0),
        assurance_education=get_float('Salaires', 'assurance_education', 50.0),
        prelevement_actuel_total=get_float('Salaires', 'prelevement_actuel_total', 0.45),

        # Taxes indirectes
        taxes_indirectes=get_taxes_indirectes(config),
    )

    return params


def get_taxes_indirectes(config) -> Dict[float, float]:
    """Extrait les taxes indirectes par tranche de salaire depuis le fichier INI"""
    taxes = {}
    if config.has_section('TaxesIndirectes'):
        for key, value in config.items('TaxesIndirectes'):
            if key.startswith('taxes_'):
                try:
                    salaire = float(key.replace('taxes_', ''))
                    taxes[salaire] = float(value)
                except ValueError:
                    pass

    # Valeurs par défaut si aucune configuration
    if not taxes:
        taxes = {
            2000: 0.16,
            3000: 0.14,
            4000: 0.12,
            5000: 0.11,
            7000: 0.09,
            10000: 0.08,
        }

    return taxes


# ==========================================
# FONCTIONS DE GÉNÉRATION GRAPHIQUES SVG
# ==========================================

def creer_svg_graphique(largeur, hauteur):
    """Crée un élément SVG de base"""
    svg = ET.Element('svg', {
        'width': str(largeur),
        'height': str(hauteur),
        'xmlns': 'http://www.w3.org/2000/svg',
        'version': '1.1'
    })
    return svg

def ajouter_grille_svg(svg, x, y, largeur, hauteur, nb_lignes=5):
    """Ajoute une grille au graphique SVG"""
    g = ET.SubElement(svg, 'g', {'class': 'grille'})
    for i in range(nb_lignes + 1):
        y_pos = y + (hauteur * i / nb_lignes)
        ET.SubElement(g, 'line', {
            'x1': str(x), 'y1': str(y_pos),
            'x2': str(x + largeur), 'y2': str(y_pos),
            'stroke': '#e0e0e0', 'stroke-width': '1'
        })

def normaliser_valeurs(valeurs, min_val, max_val):
    """Normalise les valeurs entre 0 et 1"""
    if max_val == min_val:
        return [0.5] * len(valeurs)
    return [(v - min_val) / (max_val - min_val) for v in valeurs]

def creer_chemin_svg(points_x, points_y):
    """Crée un chemin SVG à partir de points (coordonnées déjà en SVG)"""
    points = [f"{x},{y}" for x, y in zip(points_x, points_y)]
    return "M " + " L ".join(points)

def graphique_ligne_svg(svg, x, y, largeur, hauteur, donnees, champ, couleur, titre):
    """Dessine un graphique en ligne SVG"""
    # Titre
    ET.SubElement(svg, 'text', {
        'x': str(x + largeur / 2), 'y': str(y - 10),
        'text-anchor': 'middle', 'font-size': '14',
        'font-weight': 'bold', 'fill': '#333'
    }).text = titre
    
    # Grille
    ajouter_grille_svg(svg, x, y, largeur, hauteur)
    
    # Données
    valeurs = [d[champ] for d in donnees]
    if not valeurs or max(valeurs) == 0:
        return
    
    # Normalisation : TOUJOURS partir de 0
    val_min = 0
    val_max = max(valeurs)
    valeurs_norm = normaliser_valeurs(valeurs, val_min, val_max)
    
    # Points
    nb_points = len(donnees)
    points_x = [x + (largeur * i / (nb_points - 1)) for i in range(nb_points)]
    points_y = [y + (hauteur * (1 - v)) for v in valeurs_norm]
    
    # Chemin
    chemin = creer_chemin_svg(points_x, points_y)
    ET.SubElement(svg, 'path', {
        'd': chemin, 'fill': 'none',
        'stroke': couleur, 'stroke-width': '2'
    })
    
    # Zone remplie
    chemin_zone = chemin + f" L {x + largeur},{y + hauteur} L {x},{y + hauteur} Z"
    ET.SubElement(svg, 'path', {
        'd': chemin_zone, 'fill': couleur, 'fill-opacity': '0.2'
    })
    
    # Axes
    ET.SubElement(svg, 'line', {
        'x1': str(x), 'y1': str(y),
        'x2': str(x), 'y2': str(y + hauteur),
        'stroke': '#333', 'stroke-width': '2'
    })
    ET.SubElement(svg, 'line', {
        'x1': str(x), 'y1': str(y + hauteur),
        'x2': str(x + largeur), 'y2': str(y + hauteur),
        'stroke': '#333', 'stroke-width': '2'
    })
    
    # Labels min/max sur axe Y
    ET.SubElement(svg, 'text', {
        'x': str(x - 10), 'y': str(y + 5),
        'text-anchor': 'end', 'font-size': '10', 'fill': '#666'
    }).text = f"{val_max:.0f}"
    ET.SubElement(svg, 'text', {
        'x': str(x - 10), 'y': str(y + hauteur + 5),
        'text-anchor': 'end', 'font-size': '10', 'fill': '#666'
    }).text = f"{val_min:.0f}"
    
    # Labels décennies sur axe X
    duree_totale = donnees[-1]['annee'] if donnees else 80
    # Afficher tous les 10 ans
    for decennie in range(0, int(duree_totale) + 1, 10):
        if decennie < len(donnees):
            x_pos = x + (largeur * decennie / (nb_points - 1)) if nb_points > 1 else x
            # Petite marque verticale
            ET.SubElement(svg, 'line', {
                'x1': str(x_pos), 'y1': str(y + hauteur),
                'x2': str(x_pos), 'y2': str(y + hauteur + 5),
                'stroke': '#333', 'stroke-width': '1'
            })
            # Label année
            ET.SubElement(svg, 'text', {
                'x': str(x_pos), 'y': str(y + hauteur + 18),
                'text-anchor': 'middle', 'font-size': '9', 'fill': '#666'
            }).text = f"{decennie}"


# ==========================================
# CLASSE PRINCIPALE
# ==========================================

class CohorteRetraites:
    """Classe pour gérer une cohorte de retraités"""
    
    def __init__(self, annee_depart: int, nombre_initial: int, 
                 pension_annuelle: float, droits_repartition: float):
        self.annee_depart = annee_depart
        self.nombre_vivants = nombre_initial
        self.pension_annuelle = pension_annuelle
        self.droits_repartition = droits_repartition  # en fraction (ex: 0.975 pour 39/40)
    
    def appliquer_mortalite(self, taux_mortalite: float):
        """Applique la mortalité à la cohorte"""
        deces = self.nombre_vivants * taux_mortalite
        self.nombre_vivants = max(0, self.nombre_vivants - deces)
    
    def flux_pension_annuel(self) -> float:
        """Calcule le flux de pension total de cette cohorte"""
        return self.nombre_vivants * self.pension_annuelle * self.droits_repartition


class SimulateurTransition:
    """Classe principale pour simuler la transition"""
    
    def __init__(self, params: Parametres):
        self.params = params
        self.cohortes: List[CohorteRetraites] = []
        self.resultats: List[Dict] = []
        
    def initialiser_cohorte_initiale(self):
        """Initialise la cohorte des retraités existants"""
        pension_apres_reduction = (self.params.pension_moyenne_annuelle * 
                                  (1 - self.params.reduction_solidaire))
        
        cohorte = CohorteRetraites(
            annee_depart=0,
            nombre_initial=self.params.nombre_retraites_initiaux,
            pension_annuelle=pension_apres_reduction,
            droits_repartition=1.0  # 100% des droits
        )
        self.cohortes.append(cohorte)
    
    def calculer_flux_brut_pensions(self, annee: int) -> float:
        """
        Calcule le flux brut de pensions pour une année donnée
        (avant application du multiplicateur)
        Retourne le flux en milliards d'euros
        """
        flux_total = 0.0
        
        for cohorte in self.cohortes:
            if cohorte.nombre_vivants > 0:
                # Appliquer mortalité
                annees_depuis_depart = annee - cohorte.annee_depart
                taux_mort = calculer_taux_mortalite(
                    annees_depuis_depart,
                    self.params.taux_mortalite_initial,
                    self.params.increment_mortalite_annuel
                )
                cohorte.appliquer_mortalite(taux_mort)
                
                # Ajouter flux (en euros)
                flux_total += cohorte.flux_pension_annuel()
        
        # Convertir en milliards d'euros
        return flux_total / 1_000_000_000.0
    
    def calculer_multiplicateur(self, annees_simulation: int = 100) -> float:
        """
        Calcule le multiplicateur nécessaire pour que la somme des flux
        soit égale à la dette implicite
        
        Effectue une simulation complète en mode "passe 1"
        """
        # Sauvegarder l'état actuel
        cohortes_backup = [
            CohorteRetraites(c.annee_depart, c.nombre_vivants, 
                           c.pension_annuelle, c.droits_repartition)
            for c in self.cohortes
        ]
        
        # Réinitialiser
        self.cohortes = []
        self.initialiser_cohorte_initiale()
        
        somme_flux_bruts = 0.0
        pension_apres_reduction = (self.params.pension_moyenne_annuelle * 
                                  (1 - self.params.reduction_solidaire))
        
        for annee in range(1, annees_simulation + 1):
            # Ajouter nouvelle cohorte
            droits = calculer_droits_repartition(
                annee,
                self.params.duree_carriere,
                self.params.methode_droits,
                self.params.parametre_methode_droits
            )
            
            if droits > 0:
                nouvelle_cohorte = CohorteRetraites(
                    annee_depart=annee,
                    nombre_initial=self.params.nouveaux_retraites_par_an,
                    pension_annuelle=pension_apres_reduction,
                    droits_repartition=droits / self.params.duree_carriere
                )
                self.cohortes.append(nouvelle_cohorte)
            
            # Calculer flux brut
            flux = self.calculer_flux_brut_pensions(annee)
            somme_flux_bruts += flux
            
            # Arrêt si plus personne
            if all(c.nombre_vivants < 1 for c in self.cohortes):
                break
        
        # Restaurer l'état
        self.cohortes = cohortes_backup
        
        # Calculer multiplicateur
        dette_implicite_apres_reduction = (self.params.dette_implicite_initiale * 
                                          (1 - self.params.reduction_solidaire))
        
        if somme_flux_bruts == 0:
            return 1.0
        
        multiplicateur = dette_implicite_apres_reduction / somme_flux_bruts
        return multiplicateur
    
    def simuler(self) -> List[Dict]:
        """
        Effectue la simulation complète
        
        Returns:
        --------
        List[Dict] : Liste des résultats année par année
        """
        # Initialisation
        self.initialiser_cohorte_initiale()
        
        # Calculer le multiplicateur
        print("Calcul du multiplicateur...")
        multiplicateur = self.calculer_multiplicateur()
        print(f"Multiplicateur calculé: {multiplicateur:.4f}")
        
        # Réinitialiser pour la vraie simulation
        self.cohortes = []
        self.initialiser_cohorte_initiale()
        
        # Variables d'état
        pib = self.params.pib_initial
        dette_publique = self.params.dette_publique_initiale
        dette_transition = 0.0
        dette_implicite = (self.params.dette_implicite_initiale * 
                          (1 - self.params.reduction_solidaire))
        
        pension_apres_reduction = (self.params.pension_moyenne_annuelle * 
                                  (1 - self.params.reduction_solidaire))
        
        # === ENREGISTRER L'ANNÉE 0 (ÉTAT INITIAL) ===
        resultat_initial = {
            'annee': 0,
            'pib': pib,
            'differentiel_pct': 0.0,
            'differentiel_plafond_pct': self.params.differentiel_initial * 100,  # Plafond initial
            'interets_totaux': 0.0,
            'dette_publique': dette_publique,
            'dette_transition': dette_transition,
            'dette_publique_totale': dette_publique + dette_transition,
            'dette_publique_pct': (dette_publique / pib * 100) if pib > 0 else 0,
            'dette_transition_pct': (dette_transition / pib * 100) if pib > 0 else 0,
            'dette_publique_totale_pct': ((dette_publique + dette_transition) / pib * 100) if pib > 0 else 0,
            'dette_implicite': dette_implicite,
            'dette_implicite_pct': (dette_implicite / pib * 100) if pib > 0 else 0,
            'flux_pensions': 0.0,
            'nombre_retraites': self.params.nombre_retraites_initiaux,
            'retraites_equiv_repartition': self.params.nombre_retraites_initiaux,  # 100% droits à t=0
            # Valeurs initiales des mécanismes de relais (pas encore activés)
            'surplus_budgetaire': pib * self.params.surplus_budgetaire_minimal_pct_pib,
            'surplus_budgetaire_pct': self.params.surplus_budgetaire_minimal_pct_pib * 100,
            'surplus_utilise_dette_transition': 0.0,
            'remboursement_differentiel': 0.0,
            'emprunt_transition': 0.0,
        }
        self.resultats.append(resultat_initial)
        
        # Remboursement avec privatisations APRÈS l'année 0
        dette_publique -= self.params.privatisations_totales
        dette_publique = max(0, dette_publique)
        
        annee = 0
        continuer = True
        
        while continuer and annee < 150:  # Max 150 ans
            annee += 1
            
            # === 1. CROISSANCE DU PIB ===
            if annee <= 10:
                taux_croissance = (self.params.taux_croissance_base + 
                                 self.params.bonus_croissance_1_10)
            elif annee <= 20:
                taux_croissance = (self.params.taux_croissance_base + 
                                 self.params.bonus_croissance_11_20)
            else:
                taux_croissance = (self.params.taux_croissance_base + 
                                 self.params.bonus_croissance_20_plus)
            
            pib *= (1 + taux_croissance)
            
            # === 2. NOUVELLE COHORTE DE RETRAITÉS ===
            droits = calculer_droits_repartition(
                annee,
                self.params.duree_carriere,
                self.params.methode_droits,
                self.params.parametre_methode_droits
            )
            
            if droits > 0:
                nouvelle_cohorte = CohorteRetraites(
                    annee_depart=annee,
                    nombre_initial=self.params.nouveaux_retraites_par_an,
                    pension_annuelle=pension_apres_reduction,
                    droits_repartition=droits / self.params.duree_carriere
                )
                self.cohortes.append(nouvelle_cohorte)
            
            # === 3. FLUX DE PENSIONS ===
            flux_brut = self.calculer_flux_brut_pensions(annee)
            flux_pensions = flux_brut * multiplicateur  # Déjà en Mds €
            
            # === 4. DIFFÉRENTIEL (CONFORME AU MANIFESTE) ===
            # RÈGLE NORMATIVE : Le différentiel suit STRICTEMENT le plafond théorique
            # Il décroît selon la formule jusqu'à 0, sans minimum forcé

            # Manque à gagner de l'abattement : abattement × taux × travailleurs × 12 mois / 1e9
            manque_abattement = (self.params.abattement_forfaitaire *
                                self.params.taux_flat_tax *
                                self.params.nombre_travailleurs_actifs * 12) / 1e9  # En Mds €

            # Besoin total = flux pensions + manque à gagner abattement
            besoin_total = flux_pensions + manque_abattement
            
            # Calculer le plafond théorique du différentiel pour cette année
            plafond_theorique = calculer_plafond_differentiel(
                annee,
                self.params.differentiel_initial,
                self.params.duree_decroissance_differentiel,
                self.params.methode_differentiel,
                self.params.parametre_methode_differentiel
            )
            
            # Le différentiel suit STRICTEMENT le plafond théorique (pas de max avec besoins)
            differentiel_disponible = pib * plafond_theorique
            
            # Variables pour tracer le mécanisme de relais
            surplus_budgetaire = pib * self.params.surplus_budgetaire_minimal_pct_pib
            surplus_utilise_dette_transition = 0.0
            remboursement_differentiel = 0.0
            emprunt_transition = 0.0
            
            # === GESTION CONFORME AU MANIFESTE ===
            if besoin_total > differentiel_disponible:
                # CAS 1: Le différentiel est insuffisant pour couvrir les besoins
                # → Emprunter la différence (dette de transition)
                differentiel_paye = differentiel_disponible
                emprunt_transition = besoin_total - differentiel_disponible
                dette_transition += emprunt_transition
            else:
                # CAS 2: Le différentiel couvre les besoins avec un excédent
                differentiel_paye = besoin_total  # On ne prélève QUE le nécessaire
                excedent_differentiel = differentiel_disponible - besoin_total
                    
                # L'excédent sert en priorité au remboursement de la dette de transition
                if dette_transition > 0 and excedent_differentiel > 0:
                    remboursement_differentiel = min(excedent_differentiel, dette_transition)
                    dette_transition -= remboursement_differentiel
                    excedent_differentiel -= remboursement_differentiel
                    # Le différentiel payé inclut ce qui est prélevé pour rembourser
                    differentiel_paye += remboursement_differentiel
                    
                # L'excédent résiduel peut rembourser la dette publique
                if excedent_differentiel > 0 and dette_publique > 0:
                    remb_publique = min(excedent_differentiel, dette_publique)
                    dette_publique -= remb_publique
                    differentiel_paye += remb_publique

            # === 4bis. SURPLUS BUDGÉTAIRE MINIMAL (MÉCANISME DE RELAIS) ===
            # Si le différentiel est nul ou insuffisant et qu'une dette de transition subsiste,
            # le surplus budgétaire minimal constitutionnel prend le relais
            if dette_transition > 0:
                # Calculer la part du surplus disponible pour la dette de transition
                surplus_disponible_transition = (surplus_budgetaire *
                                                  self.params.surplus_max_pour_dette_transition_pct)

                # Utiliser le surplus pour rembourser la dette de transition
                surplus_utilise_dette_transition = min(surplus_disponible_transition, dette_transition)
                dette_transition -= surplus_utilise_dette_transition
            
            # === 5. REMBOURSEMENT NORMAL DETTE PUBLIQUE ===
            remboursement_normal = pib * self.params.remboursement_dette_pub_pct_pib
            dette_publique -= remboursement_normal
            dette_publique = max(0, dette_publique)
            
            # === 6. INTÉRÊTS SUR DETTES RÉELLES ===
            dette_reelle_totale = dette_publique + dette_transition
            ratio_dette = dette_reelle_totale / pib if pib > 0 else 0
            taux_interet = calculer_taux_interet(ratio_dette, 
                                                self.params.seuils_taux_interet)
            
            interets_publique = dette_publique * taux_interet
            interets_transition = dette_transition * taux_interet
            interets_totaux = interets_publique + interets_transition
            
            dette_publique += interets_publique
            dette_transition += interets_transition
            
            # === 7. MISE À JOUR DETTE IMPLICITE ===
            dette_implicite -= flux_pensions
            dette_implicite = max(0, dette_implicite)
            
            # === 8. ENREGISTREMENT RÉSULTATS ===
            resultat = {
                'annee': annee,
                'pib': pib,
                'differentiel_pct': (differentiel_paye / pib * 100) if pib > 0 else 0,
                'differentiel_plafond_pct': plafond_theorique * 100,  # Plafond constitutionnel
                'interets_totaux': interets_totaux,
                'dette_publique': dette_publique,
                'dette_transition': dette_transition,
                'dette_publique_totale': dette_publique + dette_transition,
                'dette_publique_pct': (dette_publique / pib * 100) if pib > 0 else 0,
                'dette_transition_pct': (dette_transition / pib * 100) if pib > 0 else 0,
                'dette_publique_totale_pct': ((dette_publique + dette_transition) / pib * 100) if pib > 0 else 0,
                'dette_implicite': dette_implicite,
                'dette_implicite_pct': (dette_implicite / pib * 100) if pib > 0 else 0,
                'flux_pensions': flux_pensions,
                'nombre_retraites': sum(c.nombre_vivants for c in self.cohortes),
                'retraites_equiv_repartition': sum(c.nombre_vivants * c.droits_repartition for c in self.cohortes),
                # Nouvelles données pour traçabilité du mécanisme de relais
                'surplus_budgetaire': surplus_budgetaire,
                'surplus_budgetaire_pct': self.params.surplus_budgetaire_minimal_pct_pib * 100,
                'surplus_utilise_dette_transition': surplus_utilise_dette_transition,
                'remboursement_differentiel': remboursement_differentiel,
                'emprunt_transition': emprunt_transition,
            }
            
            self.resultats.append(resultat)
            
            # === 9. CONDITION D'ARRÊT ===
            if (dette_implicite < 0.1 and 
                dette_publique < 0.1 and 
                dette_transition < 0.1):
                continuer = False
        
        return self.resultats
    
    def exporter_csv(self, fichier: str = "transition_pensions.csv"):
        """Exporte les résultats en CSV"""
        if not self.resultats:
            print("Aucun résultat à exporter")
            return
        
        colonnes = [
            'annee', 'pib', 'differentiel_pct', 'differentiel_plafond_pct', 'interets_totaux',
            'dette_publique', 'dette_transition', 'dette_publique_totale',
            'dette_publique_pct', 'dette_transition_pct', 'dette_publique_totale_pct',
            'dette_implicite', 'dette_implicite_pct',
            'flux_pensions', 'nombre_retraites', 'retraites_equiv_repartition',
            'surplus_budgetaire', 'surplus_budgetaire_pct',
            'surplus_utilise_dette_transition', 'remboursement_differentiel', 'emprunt_transition'
        ]
        
        with open(fichier, 'w', newline='', encoding='utf-8') as f:
            writer = csv.DictWriter(f, fieldnames=colonnes)
            writer.writeheader()
            writer.writerows(self.resultats)
        
        print(f"✓ Fichier CSV exporté: {fichier}")
    
    def exporter_markdown(self, fichier: str = "transition_pensions.md"):
        """Exporte un tableau détaillé année par année en Markdown"""
        if not self.resultats:
            print("Aucun résultat à exporter")
            return
        
        dec = self.params.decimales_affichage
        
        with open(fichier, 'w', encoding='utf-8') as f:
            f.write(f"# Simulation transition pensions - {self.params.nom_pays}\n\n")
            
            # Synthèse
            dernier = self.resultats[-1]
            f.write("## Synthèse\n\n")
            f.write(f"- **Durée totale**: {dernier['annee']} ans\n")
            f.write(f"- **PIB final**: {dernier['pib']:.{dec}f} Mds € ")
            f.write(f"(×{dernier['pib']/self.params.pib_initial:.2f})\n")
            f.write(f"- **Dette publique finale**: {dernier['dette_publique']:.{dec}f} Mds € ")
            f.write(f"({dernier['dette_publique_pct']:.{dec}f}% PIB)\n")
            
            # Trouver dette transition max
            dette_trans_max = max(r['dette_transition'] for r in self.resultats)
            f.write(f"- **Dette transition max**: {dette_trans_max:.{dec}f} Mds €\n\n")
            
            # Tableau détaillé ANNÉE PAR ANNÉE
            f.write("## Tableau détaillé année par année\n\n")
            f.write("| Année | PIB | Diff. | Intérêts | ")
            f.write("Dette Pub. | Dette Trans. | Dette Tot. | ")
            f.write("Dette Pub. % | Dette Trans. % | Dette Tot. % | ")
            f.write("Dette Pensions | Dette Pensions % |\n")
            
            f.write("|-------|-----|-------|----------|")
            f.write("------------|--------------|------------|")
            f.write("--------------|----------------|--------------|")
            f.write("----------------|------------------|\n")
            
            # TOUTES les années
            for r in self.resultats:
                f.write(f"| {r['annee']:3d} | {r['pib']:6.0f} | {r['differentiel_pct']:5.{dec}f} | ")
                f.write(f"{r['interets_totaux']:6.{dec}f} | {r['dette_publique']:7.0f} | ")
                f.write(f"{r['dette_transition']:7.0f} | {r['dette_publique_totale']:7.0f} | ")
                f.write(f"{r['dette_publique_pct']:6.{dec}f} | {r['dette_transition_pct']:6.{dec}f} | ")
                f.write(f"{r['dette_publique_totale_pct']:6.{dec}f} | {r['dette_implicite']:9.0f} | ")
                f.write(f"{r['dette_implicite_pct']:6.{dec}f} |\n")
        
        print(f"✓ Fichier Markdown exporté: {fichier}")
    
    def exporter_tableau_salaires(self, fichier: str = None):
        """Exporte un tableau d'analyse de l'impact sur les salaires"""
        if not self.resultats:
            print("Aucun résultat à exporter")
            return
        
        if not self.params.generer_tableau_salaires:
            return
        
        if fichier is None:
            fichier = self.params.fichier_tableau_salaires
        
        # Calcul total assurances
        assurances_total = (self.params.assurance_sante + 
                           self.params.assurance_chomage + 
                           self.params.assurance_pension + 
                           self.params.assurance_education)
        
        with open(fichier, 'w', encoding='utf-8') as f:
            f.write(f"# Impact sur les Salaires - {self.params.nom_pays}\n\n")
            f.write("## Paramètres du Nouveau Système\n\n")
            f.write(f"- **Flat tax** : {self.params.taux_flat_tax*100:.1f}%\n")
            if self.params.abattement_forfaitaire > 0:
                f.write(f"- **Abattement forfaitaire** : {self.params.abattement_forfaitaire:.0f}€/mois (non taxés)\n")
            f.write(f"- **Assurances mensuelles** :\n")
            f.write(f"  - Santé : {self.params.assurance_sante:.0f}€\n")
            f.write(f"  - Chômage : {self.params.assurance_chomage:.0f}€\n")
            f.write(f"  - Pension (capitalisation) : {self.params.assurance_pension:.0f}€\n")
            f.write(f"  - Éducation : {self.params.assurance_education:.0f}€\n")
            f.write(f"  - **Total** : {assurances_total:.0f}€/mois\n\n")
            
            f.write("## Système Actuel (Année -1)\n\n")
            f.write(f"- **Prélèvement total actuel** : {self.params.prelevement_actuel_total*100:.1f}%\n")
            f.write("  (Cotisations sociales + impôts sur le revenu)\n\n")
            
            f.write("---\n\n")
            f.write("## ✅ TOUS LES SALAIRES SONT GAGNANTS DÈS LE PREMIER JOUR\n\n")
            f.write("**Pourquoi ?** L'abolition des taxes indirectes (TVA, accises, taxes foncières) ")
            f.write("compense largement l'impact du différentiel de transition.\n\n")
            f.write("**Les taxes indirectes sont régressives** — elles pèsent plus lourd sur les bas revenus :\n")
            for salaire in self.params.salaires_analyses:
                taux_ti = self.params.taxes_indirectes.get(salaire, 0.10) * 100
                f.write(f"- Salaire {salaire:.0f}€ : {taux_ti:.0f}% du net en taxes indirectes\n")
            f.write("\n**Résultat** : Le gain relatif est plus favorable aux bas revenus (+16% vs +8%). ")
            f.write("Aucun mécanisme correctif n'est nécessaire.\n\n")
            f.write("---\n\n")
            f.write("## Tableau Comparatif (€ constants, salaires mensuels)\n\n")
            
            # En-tête du tableau
            f.write("| Année | Diff. % |")
            for salaire in self.params.salaires_analyses:
                f.write(f" {salaire:.0f}€ Net | Prél. % |")
            f.write("\n")
            
            # Ligne de séparation
            f.write("|-------|---------|")
            for _ in self.params.salaires_analyses:
                f.write("-----------|---------|")
            f.write("\n")
            
            # ANNÉE -1 : Système actuel
            f.write("|  **-1** | **Actuel** |")
            for salaire in self.params.salaires_analyses:
                net_actuel = salaire * (1 - self.params.prelevement_actuel_total)
                taux_actuel = self.params.prelevement_actuel_total * 100
                f.write(f" **{net_actuel:.0f}€** | **{taux_actuel:.1f}%** |")
            f.write("\n")
            
            # ANNÉES 0 à fin : Nouveau système
            for r in self.resultats:
                annee = r['annee']
                diff_pct = r['differentiel_pct'] / 100  # Convertir en fraction
                
                # Décalage : année 0 simulation → année -1 affichée (ignorée car = état initial)
                # année 1 simulation → année 0 affichée
                annee_affichee = annee - 1
                
                # Ignorer l'année 0 de la simulation (état initial sans différentiel)
                if annee == 0:
                    continue
                
                f.write(f"| {annee_affichee:3d} | {r['differentiel_pct']:5.2f} |")
                
                for salaire in self.params.salaires_analyses:
                    # Calcul prélèvements
                    base_imposable = max(0, salaire - self.params.abattement_forfaitaire)
                    flat_tax = base_imposable * self.params.taux_flat_tax
                    differentiel = salaire * diff_pct

                    # Salaire net SANS gain taxes indirectes
                    net_base = salaire - flat_tax - differentiel - assurances_total

                    # Gain de l'abolition des taxes indirectes
                    net_actuel = salaire * (1 - self.params.prelevement_actuel_total)
                    taux_ti = self.params.taxes_indirectes.get(salaire, 0.10)
                    gain_taxes = net_actuel * taux_ti

                    # Salaire net AVEC gain taxes indirectes
                    net = net_base + gain_taxes

                    # Taux de prélèvement effectif (en tenant compte du gain)
                    taux_prelevement = ((salaire - net) / salaire * 100) if salaire > 0 else 0

                    f.write(f" {net:6.0f}€ | {taux_prelevement:5.1f}% |")
                
                f.write("\n")
            
            # Section analyse
            f.write("\n---\n\n")
            f.write("## Analyse Comparative\n\n")
            
            # Prendre année 0 (début réforme) et dernière année
            annee_0 = next((r for r in self.resultats if r['annee'] == 1), None)
            annee_fin = self.resultats[-1]
            
            if annee_0:
                f.write("### Évolution du Prélèvement Total\n\n")
                f.write("| Période | Différentiel | Prélèvement Total |\n")
                f.write("|---------|--------------|-------------------|\n")
                f.write(f"| Année -1 (actuel) | - | {self.params.prelevement_actuel_total*100:.1f}% |\n")
                
                diff_0 = annee_0['differentiel_pct'] / 100
                taux_0 = (self.params.taux_flat_tax + diff_0 + assurances_total/self.params.salaires_analyses[2]) * 100
                f.write(f"| Année 0 (début) | {annee_0['differentiel_pct']:.1f}% | ~{taux_0:.1f}% |\n")
                
                diff_fin = annee_fin['differentiel_pct'] / 100
                taux_fin = (self.params.taux_flat_tax + diff_fin + assurances_total/self.params.salaires_analyses[2]) * 100
                f.write(f"| Année {annee_fin['annee']-1} (fin) | {annee_fin['differentiel_pct']:.1f}% | ~{taux_fin:.1f}% |\n\n")
                
                f.write(f"**Gain final** : {self.params.prelevement_actuel_total*100 - taux_fin:.1f} points\n\n")
            
            # Tableau effet combiné dès l'année 0
            f.write("### Effet Combiné dès l'Année 0 (différentiel + abolition taxes indirectes)\n\n")
            f.write("| Salaire Brut | Net Actuel | Impact diff. | Gain taxes ind. | **Effet net** |\n")
            f.write("|--------------|------------|--------------|-----------------|---------------|\n")

            annee_0 = next((r for r in self.resultats if r['annee'] == 1), None)
            if annee_0:
                diff_0_pct = annee_0['differentiel_pct'] / 100
                for salaire in self.params.salaires_analyses:
                    net_actuel = salaire * (1 - self.params.prelevement_actuel_total)

                    # Impact du différentiel seul
                    base_imposable = max(0, salaire - self.params.abattement_forfaitaire)
                    flat_tax = base_imposable * self.params.taux_flat_tax
                    differentiel = salaire * diff_0_pct
                    net_diff_seul = salaire - flat_tax - differentiel - assurances_total
                    impact_diff = net_diff_seul - net_actuel

                    # Gain taxes indirectes
                    taux_ti = self.params.taxes_indirectes.get(salaire, 0.10)
                    gain_taxes = net_actuel * taux_ti

                    # Effet net
                    effet_net = impact_diff + gain_taxes

                    signe_diff = "+" if impact_diff >= 0 else ""
                    f.write(f"| {salaire:.0f}€ | {net_actuel:.0f}€ | {signe_diff}{impact_diff:.0f}€ | +{gain_taxes:.0f}€ | **+{effet_net:.0f}€** ✓ |\n")

            f.write("\n**Tous les salaires sont gagnants dès le premier jour !**\n\n")

            f.write("### Gain de Pouvoir d'Achat (Année -1 → Année Finale)\n\n")
            f.write("| Salaire Brut | Net Actuel | Net Final | Gain €/mois | Gain % |\n")
            f.write("|--------------|------------|-----------|-------------|--------|\n")

            diff_fin_pct = self.resultats[-1]['differentiel_pct'] / 100
            for salaire in self.params.salaires_analyses:
                net_actuel = salaire * (1 - self.params.prelevement_actuel_total)

                base_imposable = max(0, salaire - self.params.abattement_forfaitaire)
                flat_tax = base_imposable * self.params.taux_flat_tax
                differentiel = salaire * diff_fin_pct
                net_base = salaire - flat_tax - differentiel - assurances_total

                # Gain taxes indirectes (permanent)
                taux_ti = self.params.taxes_indirectes.get(salaire, 0.10)
                gain_taxes = net_actuel * taux_ti
                net_final = net_base + gain_taxes

                gain_euros = net_final - net_actuel
                gain_pct = (gain_euros / net_actuel * 100) if net_actuel > 0 else 0

                f.write(f"| {salaire:.0f}€ | {net_actuel:.0f}€ | {net_final:.0f}€ | +{gain_euros:.0f}€ | +{gain_pct:.1f}% |\n")
        
        print(f"✓ Tableau salaires exporté: {fichier}")
    
    def exporter_svg(self, fichier: str = "transition_pensions_graphiques.svg"):
        """
        Exporte les graphiques au format SVG
        
        Parameters:
        -----------
        fichier : str
            Nom du fichier de sortie
        """
        if not self.resultats:
            print("⚠ Aucun résultat à exporter")
            return
        
        # Dimensions
        largeur_totale = 1200
        hauteur_totale = 900
        marge = 80
        espacement = 100
        
        largeur_graph = (largeur_totale - 3 * marge) // 2
        hauteur_graph = (hauteur_totale - 4 * espacement) // 3
        
        # Créer SVG principal
        svg = creer_svg_graphique(largeur_totale, hauteur_totale)
        
        # Fond blanc
        ET.SubElement(svg, 'rect', {
            'width': str(largeur_totale),
            'height': str(hauteur_totale),
            'fill': 'white'
        })
        
        # Titre principal
        ET.SubElement(svg, 'text', {
            'x': str(largeur_totale / 2),
            'y': '30',
            'text-anchor': 'middle',
            'font-size': '18',
            'font-weight': 'bold',
            'fill': '#333'
        }).text = f'Simulation Transition Système de Pensions - {self.params.nom_pays}'
        
        # Graphique 1: PIB
        graphique_ligne_svg(
            svg, marge, espacement,
            largeur_graph, hauteur_graph,
            self.resultats, 'pib', '#4A90E2',
            'Évolution du PIB (Mds €)'
        )
        
        # Graphique 2: Différentiel
        graphique_ligne_svg(
            svg, marge + largeur_graph + marge, espacement,
            largeur_graph, hauteur_graph,
            self.resultats, 'differentiel_pct', '#E85D75',
            'Différentiel (% du PIB)'
        )
        
        # Graphique 3: Dette publique totale
        graphique_ligne_svg(
            svg, marge, espacement * 2 + hauteur_graph,
            largeur_graph, hauteur_graph,
            self.resultats, 'dette_publique_totale_pct', '#50C878',
            'Dette Publique Totale (% PIB)'
        )
        
        # Graphique 4: Dette implicite
        graphique_ligne_svg(
            svg, marge + largeur_graph + marge, espacement * 2 + hauteur_graph,
            largeur_graph, hauteur_graph,
            self.resultats, 'dette_implicite_pct', '#9B59B6',
            'Dette Implicite Pensions (% PIB)'
        )
        
        # Graphique 5: Intérêts
        graphique_ligne_svg(
            svg, marge, espacement * 3 + hauteur_graph * 2,
            largeur_graph, hauteur_graph,
            self.resultats, 'interets_totaux', '#F39C12',
            'Intérêts sur Dettes Réelles (Mds €)'
        )
        
        # Graphique 6: Comparaison dettes (avec 2 lignes)
        x6 = marge + largeur_graph + marge
        y6 = espacement * 3 + hauteur_graph * 2
        
        ET.SubElement(svg, 'text', {
            'x': str(x6 + largeur_graph / 2),
            'y': str(y6 - 10),
            'text-anchor': 'middle',
            'font-size': '14',
            'font-weight': 'bold',
            'fill': '#333'
        }).text = 'Comparaison Dettes (% PIB)'
        
        ajouter_grille_svg(svg, x6, y6, largeur_graph, hauteur_graph)
        
        # Dette réelle
        valeurs_reelle = [d['dette_publique_totale_pct'] for d in self.resultats]
        valeurs_implicite = [d['dette_implicite_pct'] for d in self.resultats]
        val_max = max(max(valeurs_reelle), max(valeurs_implicite))
        valeurs_norm_reelle = normaliser_valeurs(valeurs_reelle, 0, val_max)
        
        nb_points = len(self.resultats)
        points_x = [x6 + (largeur_graph * i / (nb_points - 1)) for i in range(nb_points)]
        points_y_reelle = [y6 + (hauteur_graph * (1 - v)) for v in valeurs_norm_reelle]
        
        chemin_reelle = creer_chemin_svg(points_x, points_y_reelle)
        ET.SubElement(svg, 'path', {
            'd': chemin_reelle,
            'fill': 'none',
            'stroke': '#50C878',
            'stroke-width': '2'
        })
        
        # Dette implicite
        valeurs_norm_implicite = normaliser_valeurs(valeurs_implicite, 0, val_max)
        points_y_implicite = [y6 + (hauteur_graph * (1 - v)) for v in valeurs_norm_implicite]
        
        chemin_implicite = creer_chemin_svg(points_x, points_y_implicite)
        ET.SubElement(svg, 'path', {
            'd': chemin_implicite,
            'fill': 'none',
            'stroke': '#9B59B6',
            'stroke-width': '2',
            'stroke-dasharray': '5,5'
        })
        
        # Légende
        leg_x = x6 + largeur_graph - 150
        leg_y = y6 + 20
        
        ET.SubElement(svg, 'line', {
            'x1': str(leg_x), 'y1': str(leg_y),
            'x2': str(leg_x + 30), 'y2': str(leg_y),
            'stroke': '#50C878', 'stroke-width': '2'
        })
        ET.SubElement(svg, 'text', {
            'x': str(leg_x + 35), 'y': str(leg_y + 4),
            'font-size': '10', 'fill': '#666'
        }).text = 'Dette réelle'
        
        ET.SubElement(svg, 'line', {
            'x1': str(leg_x), 'y1': str(leg_y + 15),
            'x2': str(leg_x + 30), 'y2': str(leg_y + 15),
            'stroke': '#9B59B6', 'stroke-width': '2', 'stroke-dasharray': '5,5'
        })
        ET.SubElement(svg, 'text', {
            'x': str(leg_x + 35), 'y': str(leg_y + 19),
            'font-size': '10', 'fill': '#666'
        }).text = 'Dette implicite'
        
        # Axes pour graphique 6
        ET.SubElement(svg, 'line', {
            'x1': str(x6), 'y1': str(y6),
            'x2': str(x6), 'y2': str(y6 + hauteur_graph),
            'stroke': '#333', 'stroke-width': '2'
        })
        ET.SubElement(svg, 'line', {
            'x1': str(x6), 'y1': str(y6 + hauteur_graph),
            'x2': str(x6 + largeur_graph), 'y2': str(y6 + hauteur_graph),
            'stroke': '#333', 'stroke-width': '2'
        })
        
        # Labels min/max sur axe Y
        ET.SubElement(svg, 'text', {
            'x': str(x6 - 10), 'y': str(y6 + 5),
            'text-anchor': 'end', 'font-size': '10', 'fill': '#666'
        }).text = f"{val_max:.0f}"
        ET.SubElement(svg, 'text', {
            'x': str(x6 - 10), 'y': str(y6 + hauteur_graph + 5),
            'text-anchor': 'end', 'font-size': '10', 'fill': '#666'
        }).text = "0"
        
        # Labels décennies sur axe X
        duree_totale = self.resultats[-1]['annee'] if self.resultats else 80
        for decennie in range(0, int(duree_totale) + 1, 10):
            if decennie < len(self.resultats):
                x_pos = x6 + (largeur_graph * decennie / (nb_points - 1)) if nb_points > 1 else x6
                # Petite marque verticale
                ET.SubElement(svg, 'line', {
                    'x1': str(x_pos), 'y1': str(y6 + hauteur_graph),
                    'x2': str(x_pos), 'y2': str(y6 + hauteur_graph + 5),
                    'stroke': '#333', 'stroke-width': '1'
                })
                # Label année
                ET.SubElement(svg, 'text', {
                    'x': str(x_pos), 'y': str(y6 + hauteur_graph + 18),
                    'text-anchor': 'middle', 'font-size': '9', 'fill': '#666'
                }).text = f"{decennie}"
        
        # Écrire le fichier
        tree = ET.ElementTree(svg)
        ET.indent(tree, space="  ")
        tree.write(fichier, encoding='utf-8', xml_declaration=True)
        print(f"✓ Graphiques SVG exportés: {fichier}")


def main():
    """Fonction principale"""
    print("=" * 60)
    print("SIMULATION TRANSITION SYSTÈME DE PENSIONS")
    print("=" * 60)
    print()
    
    # Charger la configuration
    nom_scenario = "defaut"
    if len(sys.argv) > 1:
        fichier_ini = sys.argv[1]
        # Extraire le nom du scénario (sans chemin ni extension)
        import os
        nom_scenario = os.path.splitext(os.path.basename(fichier_ini))[0]
        
        print(f"Chargement de la configuration depuis: {fichier_ini}")
        print(f"Scénario: {nom_scenario}")
        try:
            params = charger_configuration(fichier_ini)
            print(f"✓ Configuration chargée avec succès")
        except Exception as e:
            print(f"✗ Erreur lors du chargement du fichier INI: {e}")
            print("Utilisation des paramètres par défaut")
            params = Parametres()
    else:
        print("Aucun fichier INI fourni, utilisation des paramètres par défaut")
        print("Usage: python transition_pensions.py [fichier.ini]")
        params = Parametres()
    
    print()
    print(f"Pays: {params.nom_pays}")
    print(f"PIB initial: {params.pib_initial} Mds €")
    print(f"Dette publique: {params.dette_publique_initiale} Mds €")
    print(f"Dette implicite: {params.dette_implicite_initiale} Mds €")
    print(f"Privatisations: {params.privatisations_totales} Mds €")
    print()
    
    # Créer le dossier de sortie
    import os
    dossier_sortie = f"/mnt/user-data/outputs/resultats/{nom_scenario}"
    os.makedirs(dossier_sortie, exist_ok=True)
    print(f"Dossier de sortie: {dossier_sortie}")
    print()
    
    # Créer simulateur et lancer
    sim = SimulateurTransition(params)
    
    print("Lancement de la simulation...")
    print()
    
    resultats = sim.simuler()
    
    print()
    print(f"✓ Simulation terminée: {len(resultats)} années")
    print()
    
    # Exporter dans le sous-dossier
    sim.exporter_csv(f"{dossier_sortie}/donnees.csv")
    sim.exporter_markdown(f"{dossier_sortie}/tableaux.md")
    sim.exporter_tableau_salaires(f"{dossier_sortie}/salaires.md")
    sim.exporter_svg(f"{dossier_sortie}/graphiques.svg")
    
    print()
    print("=" * 60)
    print("SIMULATION TERMINÉE")
    print("=" * 60)


if __name__ == "__main__":
    main()
