#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Module de mise à l'échelle automatique pour écrans haute résolution (4K)

Ce module détecte automatiquement la résolution et le DPI de l'écran,
puis applique un facteur de scaling approprié à :
- Tkinter (widgets, polices)
- Matplotlib (figures, polices, ticks)

Usage simple:
    import scaling
    scaling.auto_scaling()

Usage avec facteur manuel:
    import scaling
    scaling.apply_scaling(1.5)  # 150%

Compatible Python 3.8+, Linux, WSL, Windows, macOS
"""

import os
import sys
from typing import Optional, Tuple

# Facteur de scaling global
CURRENT_SCALE = 1.0

# DPI de référence (96 DPI = 100%)
REFERENCE_DPI = 96


def detect_display_info() -> Tuple[int, int, int, float]:
    """
    Détecte les informations d'affichage.

    Returns:
        Tuple (largeur, hauteur, dpi, scale_factor)
    """
    width, height, dpi = 1920, 1080, 96
    scale_factor = 1.0

    try:
        # Méthode 1: Tkinter (plus fiable)
        import tkinter as tk
        root = tk.Tk()
        root.withdraw()

        width = root.winfo_screenwidth()
        height = root.winfo_screenheight()

        # Essayer d'obtenir le DPI via Tk
        try:
            dpi = root.winfo_fpixels('1i')
        except:
            # Fallback: utiliser la résolution pour estimer le DPI
            # Écran 4K (3840x2160) typiquement à 27" = ~163 DPI
            if width >= 3840:
                dpi = 163
            elif width >= 2560:
                dpi = 109
            else:
                dpi = 96

        root.destroy()

    except Exception as e:
        print(f"[scaling] Détection Tkinter échouée: {e}")

        # Méthode 2: Variables d'environnement
        try:
            if 'GDK_SCALE' in os.environ:
                scale_factor = float(os.environ['GDK_SCALE'])
            elif 'QT_SCALE_FACTOR' in os.environ:
                scale_factor = float(os.environ['QT_SCALE_FACTOR'])
        except:
            pass

    # Calculer le facteur de scaling basé sur le DPI
    if scale_factor == 1.0:
        scale_factor = dpi / REFERENCE_DPI

    # Arrondir à des valeurs pratiques (1.0, 1.25, 1.5, 1.75, 2.0, etc.)
    scale_factor = round(scale_factor * 4) / 4

    # Limiter à des valeurs raisonnables
    scale_factor = max(1.0, min(3.0, scale_factor))

    return width, height, int(dpi), scale_factor


def detect_scaling() -> float:
    """
    Détecte automatiquement le facteur de scaling recommandé.

    Returns:
        float: Facteur de scaling (1.0 = 100%, 2.0 = 200%)
    """
    _, _, _, scale = detect_display_info()
    return scale


def apply_scaling_tkinter(scale: float):
    """
    Applique le scaling à Tkinter.

    Args:
        scale: Facteur de scaling
    """
    try:
        import tkinter as tk
        from tkinter import ttk, font

        # Créer une fenêtre temporaire pour configurer
        root = tk.Tk()
        root.withdraw()

        # Activer la gestion DPI sur Windows
        if sys.platform == 'win32':
            try:
                from ctypes import windll
                windll.shcore.SetProcessDpiAwareness(1)
            except:
                pass

        # Configurer le scaling Tk
        root.tk.call('tk', 'scaling', scale)

        # Obtenir les polices par défaut et les agrandir
        default_font = font.nametofont("TkDefaultFont")
        default_size = default_font.cget("size")
        new_size = int(default_size * scale) if default_size > 0 else int(10 * scale)
        default_font.configure(size=new_size)

        text_font = font.nametofont("TkTextFont")
        text_size = text_font.cget("size")
        new_text_size = int(text_size * scale) if text_size > 0 else int(10 * scale)
        text_font.configure(size=new_text_size)

        fixed_font = font.nametofont("TkFixedFont")
        fixed_size = fixed_font.cget("size")
        new_fixed_size = int(fixed_size * scale) if fixed_size > 0 else int(10 * scale)
        fixed_font.configure(size=new_fixed_size)

        root.destroy()

    except Exception as e:
        print(f"[scaling] Configuration Tkinter échouée: {e}")


def apply_scaling_matplotlib(scale: float):
    """
    Applique le scaling à Matplotlib.

    Args:
        scale: Facteur de scaling
    """
    try:
        import matplotlib.pyplot as plt
        import matplotlib as mpl

        # DPI pour les figures
        base_dpi = 100
        scaled_dpi = int(base_dpi * scale)

        # Tailles de police de base
        base_font_size = 10

        # Configuration rcParams
        params = {
            'figure.dpi': scaled_dpi,
            'savefig.dpi': scaled_dpi,

            # Polices - mise à l'échelle
            'font.size': base_font_size * scale,
            'axes.titlesize': 12 * scale,
            'axes.labelsize': 10 * scale,
            'xtick.labelsize': 9 * scale,
            'ytick.labelsize': 9 * scale,
            'legend.fontsize': 9 * scale,
            'legend.title_fontsize': 10 * scale,

            # Épaisseurs de lignes
            'lines.linewidth': 1.5 * scale,
            'axes.linewidth': 0.8 * scale,
            'grid.linewidth': 0.5 * scale,
            'xtick.major.width': 0.8 * scale,
            'ytick.major.width': 0.8 * scale,
            'xtick.minor.width': 0.6 * scale,
            'ytick.minor.width': 0.6 * scale,

            # Tailles des marqueurs de ticks
            'xtick.major.size': 4 * scale,
            'ytick.major.size': 4 * scale,
            'xtick.minor.size': 2 * scale,
            'ytick.minor.size': 2 * scale,

            # Marges et padding
            'axes.titlepad': 6 * scale,
            'axes.labelpad': 4 * scale,
        }

        plt.rcParams.update(params)

    except ImportError:
        print("[scaling] Matplotlib non disponible")
    except Exception as e:
        print(f"[scaling] Configuration Matplotlib échouée: {e}")


def apply_scaling(scale: float = None):
    """
    Applique un facteur de scaling à tous les composants.

    Args:
        scale: Facteur de scaling (si None, utilise auto-détection)
    """
    global CURRENT_SCALE

    if scale is None:
        scale = detect_scaling()

    # Limiter aux valeurs raisonnables
    scale = max(1.0, min(3.0, scale))
    CURRENT_SCALE = scale

    print(f"[scaling] Application du facteur de scaling: {scale:.2f}x")

    # Appliquer aux différents systèmes
    apply_scaling_tkinter(scale)
    apply_scaling_matplotlib(scale)

    return scale


def auto_scaling():
    """
    Détecte et applique automatiquement le scaling optimal.

    Returns:
        float: Le facteur de scaling appliqué
    """
    # Vérifier d'abord les variables d'environnement pour un override manuel
    env_scale = os.environ.get('SIMUL_SCALE')
    if env_scale:
        try:
            scale = float(env_scale)
            print(f"[scaling] Utilisation du scale depuis SIMUL_SCALE={scale}")
            return apply_scaling(scale)
        except ValueError:
            pass

    # Auto-détection
    width, height, dpi, scale = detect_display_info()
    print(f"[scaling] Écran détecté: {width}x{height}, DPI={dpi}, scale recommandé={scale:.2f}x")

    return apply_scaling(scale)


def get_current_scale() -> float:
    """Retourne le facteur de scaling actuellement appliqué."""
    return CURRENT_SCALE


def get_scaled_font_size(base_size: int) -> int:
    """
    Calcule une taille de police mise à l'échelle.

    Args:
        base_size: Taille de base

    Returns:
        Taille mise à l'échelle
    """
    return int(base_size * CURRENT_SCALE)


def get_scaled_dimension(base_dim: int) -> int:
    """
    Calcule une dimension mise à l'échelle.

    Args:
        base_dim: Dimension de base

    Returns:
        Dimension mise à l'échelle
    """
    return int(base_dim * CURRENT_SCALE)


# Point d'entrée pour test direct
if __name__ == "__main__":
    print("=== Test du module scaling ===")
    print()

    width, height, dpi, scale = detect_display_info()
    print(f"Résolution: {width}x{height}")
    print(f"DPI: {dpi}")
    print(f"Facteur recommandé: {scale:.2f}x")
    print()

    print("Application du scaling...")
    applied_scale = auto_scaling()
    print(f"Scaling appliqué: {applied_scale:.2f}x")
    print()

    print("Exemples de dimensions mises à l'échelle:")
    for size in [8, 10, 12, 14, 16]:
        print(f"  Police {size}pt -> {get_scaled_font_size(size)}pt")
    for dim in [100, 200, 400, 800]:
        print(f"  Dimension {dim}px -> {get_scaled_dimension(dim)}px")
