# Documentation - Simulateur de Transition Pensions

## Vue d'ensemble

Ce simulateur permet de modéliser la transition d'un système de pensions par répartition vers un système par capitalisation pour différents pays.

## Fichiers générés

- **transition_pensions.py** : Script Python principal
- **transition_pensions.csv** : Données détaillées année par année
- **transition_pensions.md** : Tableau synthétique
- **README_UTILISATION.md** : Ce fichier

## Installation et exécution

```bash
python transition_pensions.py
```

Aucune dépendance externe n'est requise (utilise uniquement les bibliothèques standard Python).

## Configuration des paramètres

Tous les paramètres sont configurables dans la classe `Parametres` :

### Paramètres économiques

```python
pib_initial: float = 850.0  # PIB initial en milliards €
dette_publique_initiale: float = 884.0  # Dette publique en milliards €

# Taux de croissance du PIB
taux_croissance_base: float = 0.015  # 1.5%/an
bonus_croissance_1_10: float = 0.02  # +2% années 1-10
bonus_croissance_11_20: float = 0.01  # +1% années 11-20
bonus_croissance_20_plus: float = 0.005  # +0.5% années 20+
```

### Paramètres du système de pensions

```python
dette_implicite_initiale: float = 2100.0  # Milliards € (avant réduction)
reduction_solidaire: float = 0.10  # 10% de réduction progressive

nombre_retraites_initiaux: int = 2_500_000
pension_moyenne_annuelle: float = 15_000.0  # €/an
nouveaux_retraites_par_an: int = 100_000

duree_carriere: int = 40  # années
age_depart_retraite: int = 65  # ans
age_esperance_vie: int = 85  # ans
```

### Paramètres de mortalité

```python
taux_mortalite_initial: float = 0.05  # 5%/an
increment_mortalite_annuel: float = 0.003  # +0.3%/an
```

La mortalité augmente progressivement :
- Année 0 de retraite : 5.0%
- Année 10 : 8.0%
- Année 20 : 11.0%

### Paramètres des droits à répartition

Trois méthodes disponibles :

#### 1. Quadratique (par défaut)

```python
methode_droits: str = 'quadratique'
parametre_methode_droits: float = 1.5  # Exposant
```

Valeurs de l'exposant :
- **1.3** : Décroissance très douce
- **1.5** : Décroissance douce (défaut)
- **2.0** : Décroissance agressive

Résultat avec exposant 1.5 :
- Année 1 : 40.00/40 (100%)
- Année 2 : 39.75/40 (99.4%)
- Année 10 : 37.38/40 (93.5%)
- Année 20 : 22.79/40 (57.0%)
- Année 40 : 0/40 (0%)

#### 2. Exponentielle

```python
methode_droits: str = 'exponentielle'
parametre_methode_droits: float = 0.05  # Taux de décroissance
```

Valeurs du taux :
- **0.03** : Très douce
- **0.05** : Moyenne (défaut)
- **0.08** : Agressive

#### 3. Par paliers

```python
methode_droits: str = 'paliers'
parametre_methode_droits = [
    (5, 0.20),   # Années 1-5 : -0.2/an
    (15, 0.50),  # Années 6-15 : -0.5/an
    (25, 1.00),  # Années 16-25 : -1.0/an
    (40, 1.60)   # Années 26-40 : -1.6/an
]
```

Format : liste de tuples `(année_fin_palier, taux_perte_annuel)`

### Paramètres de financement

```python
privatisations_totales: float = 195.0  # Milliards €
differentiel_max_pct_pib: float = 0.10  # 10% du PIB
remboursement_dette_pub_pct_pib: float = 0.01  # 1% du PIB/an
```

**Important** : Les privatisations sont utilisées **immédiatement** en année 1 pour rembourser la dette publique, ce qui réduit les intérêts futurs.

### Paramètres des taux d'intérêt

```python
seuils_taux_interet = [
    (0.60, 0.015, 0.020),   # ratio < 60% : taux 1.5-2%
    (0.90, 0.020, 0.025),   # ratio 60-90% : taux 2-2.5%
    (1.20, 0.025, 0.030),   # ratio 90-120% : taux 2.5-3%
    (9.99, 0.030, 0.040),   # ratio > 120% : taux 3-4%
]
```

Le taux d'intérêt dépend du ratio : **(dette_publique + dette_transition) / PIB**

## Les 3 types de dettes

### 1. Dette publique (dette réelle)
- Dette existante de l'État
- **Génère des intérêts** chaque année
- Diminue via remboursement normal (1% PIB/an) + excédent différentiel
- Ne peut jamais être négative

### 2. Dette de transition (dette réelle créée)
- Créée quand flux pensions > 10% PIB
- **Génère des intérêts** chaque année
- Remboursée en priorité avec excédent différentiel
- Ne peut jamais être négative

### 3. Dette implicite pensions (dette nominale)
- Engagement de l'État envers futurs retraités
- **NE génère PAS d'intérêts** (c'est une promesse, pas une dette bancaire)
- Diminue uniquement quand on paie les flux de pensions
- Descend à 0 quand le dernier retraité meurt

## Logique du différentiel

Le différentiel est un impôt temporaire pour financer la transition :

### Phase 1 : Différentiel élevé (années 1-20)
- Flux pensions très élevés
- Différentiel plafonné à 10% du PIB
- Si flux > 10% PIB → on emprunte la différence (dette transition)
- Si flux < 10% PIB → on paie quand même 10%, l'excédent rembourse les dettes

### Phase 2 : Dettes réelles remboursées (années 15-25)
- Une fois dette_publique ET dette_transition = 0
- On arrête de payer 10%
- On paie seulement le flux de pensions réel
- Le différentiel descend progressivement à 0

### Phase 3 : Extinction (années 25-76)
- Flux de pensions diminue (retraités meurent)
- Nouveaux retraités ont de moins en moins de droits
- Différentiel → 0

## Le multiplicateur

**Problème** : La somme des flux bruts de pensions ne suffit pas à rembourser la dette implicite.

**Solution** : Calculer un multiplicateur

```
multiplicateur = dette_implicite / somme_flux_bruts
flux_réel = flux_brut × multiplicateur
```

**Pour la Belgique** : multiplicateur ≈ 2.9

Cela signifie que chaque pension est environ 2.9× plus élevée que ce qu'elle serait sans ajustement, pour garantir que la dette implicite soit remboursée exactement.

## Ordre des calculs (chaque année)

1. **Croissance du PIB** (selon taux + bonus)
2. **Nouvelle cohorte de retraités** avec droits proportionnels
3. **Flux de pensions** (avec mortalité + multiplicateur)
4. **Différentiel** (plafonné à 10% ou flux réel)
5. **Remboursement dette transition** (priorité)
6. **Remboursement dette publique** (normal + excédent)
7. **Calcul intérêts** sur dettes réelles
8. **Mise à jour dette implicite** (diminue du flux payé)

## Exemple de configuration pour un autre pays

### France

```python
params = Parametres()
params.nom_pays = "France"
params.pib_initial = 2800.0  # Mds €
params.dette_publique_initiale = 3200.0  # Mds €
params.dette_implicite_initiale = 7000.0  # Mds €
params.nombre_retraites_initiaux = 17_000_000
params.pension_moyenne_annuelle = 14_000.0
params.nouveaux_retraites_par_an = 800_000
params.privatisations_totales = 300.0  # Mds €

sim = SimulateurTransition(params)
sim.simuler()
sim.exporter_csv("transition_france.csv")
sim.exporter_markdown("transition_france.md")
```

### Allemagne

```python
params = Parametres()
params.nom_pays = "Allemagne"
params.pib_initial = 4100.0  # Mds €
params.dette_publique_initiale = 2800.0  # Mds €
params.dette_implicite_initiale = 9500.0  # Mds €
params.nombre_retraites_initiaux = 21_000_000
params.pension_moyenne_annuelle = 16_500.0
params.nouveaux_retraites_par_an = 900_000
params.privatisations_totales = 500.0  # Mds €

# Allemagne : mortalité légèrement plus faible
params.taux_mortalite_initial = 0.045
params.age_esperance_vie = 86

sim = SimulateurTransition(params)
sim.simuler()
sim.exporter_csv("transition_allemagne.csv")
sim.exporter_markdown("transition_allemagne.md")
```

## Résultats attendus

### Fichier CSV

Contient toutes les années avec les colonnes :
- `annee` : Année de simulation
- `pib` : PIB en milliards €
- `differentiel_pct` : Différentiel en % du PIB
- `interets_totaux` : Intérêts payés en milliards €
- `dette_publique` : Dette publique en milliards €
- `dette_transition` : Dette transition en milliards €
- `dette_publique_totale` : Somme des 2 dettes réelles
- `dette_publique_pct`, `dette_transition_pct`, `dette_publique_totale_pct` : En % du PIB
- `dette_implicite` : Dette implicite en milliards €
- `dette_implicite_pct` : En % du PIB
- `flux_pensions` : Flux de pensions payé
- `nombre_retraites` : Nombre de retraités vivants

### Fichier Markdown

Tableau synthétique avec données toutes les 5 ans + synthèse finale.

## Validation des résultats

Le script vérifie automatiquement :

1. ✅ Dette publique ≥ 0 (jamais négative)
2. ✅ Dette transition ≥ 0 (jamais négative)
3. ✅ Dette implicite → 0 à la fin
4. ✅ Différentiel = 10% au début (quand flux élevés)
5. ✅ Différentiel diminue progressivement
6. ✅ Différentiel → 0 quand plus de flux

## Résultats pour la Belgique (défaut)

- **Durée totale** : 76 ans
- **PIB final** : 4 652 Mds € (×5.47)
- **Dette publique finale** : 0 Mds €
- **Dette transition max** : 12.7 Mds € (année 4)
- **Différentiel max** : 10% du PIB (années 1-17)

### Évolution du différentiel
- Années 1-17 : 10.0% (plafonné)
- Année 18 : 3.3%
- Année 21 : 2.5%
- Année 31 : 1.0%
- Année 41 : 0.3%
- Année 76 : 0.0%

## Questions fréquentes

### Pourquoi le multiplicateur est-il > 1 ?

La dette implicite (1 890 Mds €) est très élevée par rapport aux flux bruts de pensions naturels. Le multiplicateur ajuste proportionnellement tous les flux pour garantir que la somme totale = dette implicite.

### Pourquoi la dette implicite ne diminue-t-elle pas plus vite ?

La dette implicite diminue uniquement quand on paie les flux de pensions. Au début, beaucoup de retraités → flux élevés → diminution rapide. Puis moins de retraités → diminution plus lente.

### Pourquoi créer une dette de transition ?

Quand le flux de pensions dépasse 10% du PIB, on ne peut pas demander aux citoyens de payer plus. On emprunte temporairement la différence, puis on la rembourse quand les flux baissent.

### Les privatisations sont-elles étalées sur 20 ans ?

Non, elles sont utilisées **immédiatement** en année 1 pour rembourser la dette publique. Cela réduit massivement les intérêts futurs (économie de ~3-4 Mds €/an).

## Modifications avancées

### Changer la fonction de mortalité

Modifier la fonction `calculer_taux_mortalite()` pour utiliser une courbe plus sophistiquée :

```python
def calculer_taux_mortalite(annee_depuis_retraite, taux_initial, increment_annuel):
    # Exemple : courbe sigmoïde
    import math
    t = annee_depuis_retraite
    return 0.15 / (1 + math.exp(-0.3 * (t - 15)))
```

### Changer les paliers de droits

```python
params.methode_droits = 'paliers'
params.parametre_methode_droits = [
    (3, 0.10),    # Très douce au début
    (10, 0.30),   
    (20, 0.80),   
    (35, 1.50),   
    (45, 2.00)    # Accélération à la fin
]
```

### Ajouter des graphiques

```python
import matplotlib.pyplot as plt

# Récupérer les résultats
annees = [r['annee'] for r in sim.resultats]
dette_totale_pct = [r['dette_publique_totale_pct'] for r in sim.resultats]
differentiel = [r['differentiel_pct'] for r in sim.resultats]

plt.figure(figsize=(12, 6))
plt.subplot(1, 2, 1)
plt.plot(annees, dette_totale_pct)
plt.title('Dette publique totale (% PIB)')
plt.xlabel('Année')
plt.ylabel('%')

plt.subplot(1, 2, 2)
plt.plot(annees, differentiel)
plt.title('Différentiel (% PIB)')
plt.xlabel('Année')
plt.ylabel('%')

plt.tight_layout()
plt.savefig('graphiques_transition.png')
```

## Licence

Ce script est fourni tel quel pour analyse et simulation de politiques publiques.
